import random

from lxml import etree

from xmltrans.handlers import ElementHandler
from xmltrans.utils import node_copy


__all__ = ['IgnoreHandler', 'RandomSelectionHandler', 'CopyHandler']


class IgnoreHandler(ElementHandler):
    """The simplest case: handler always ignores given tag.

    :param node: Node to be processed.
    :returns: :class:`NoneType`

    """    
    def _process(self, node):
        return None


class RandomSelectionHandler(ElementHandler):
    """Returns a tuple composed of a processing result (:class:`etree.Element`
    instance or value) and randomly selected node from the original tree.

    :param node: Node to be processed.
    :returns: :class:`tuple`\(:class:`etree.Element`, :class:`etree.Element`)

    """    
    def _process(self, node):
        children = node.getchildren()
        tries_list = []
        non_empty_node = None
        while len(tries_list) < len(children):
            selected_node = children[random.randint(0, len(children)-1)]
            if id(selected_node) not in tries_list:
                tries_list.append(id(selected_node))
            node_is_empty = self.factory.is_node_empty(selected_node)
            if node_is_empty:
                break
        return self.factory.node_process(selected_node), selected_node
    

class CopyHandler(ElementHandler):
    """Returns new node instance with ``attrib``, ``text`` and ``tail``
    attributes copied and with original tag name preserved.

    :param node: Node to be processed.
    :returns: :class:`etree.Element`

    """    
    def _process(self, node):
        new_node = node_copy(node)
        return new_node