import copy

from lxml import etree

from xmltrans.exceptions import HandlerMissing
from xmltrans.exceptions import ImproperlyConfigured
from xmltrans.handlers import ElementHandler
from xmltrans.utils import clean_tag_name
from xmltrans.utils import node_copy


class ParentStack(object):

    def __init__(self, factory):
        self.factory = factory

    def last(self):
        """Returns latest parent from the stack without removing it.

        :returns: :class:`etree.Element`

        """
        if self.factory.parent_stack:
            return self.factory.parent_stack[-1]
        return None

    def pop(self):
        """Removes last parent from the stack.

        :returns: :class:`etree.Element`

        """
        if self.factory.parent_stack is not None:
            return self.factory.parent_stack.pop()

    def add(self, node):
        """Adds given node to the stack, so it may serve as a parent for
        other nodes.

        :param node: node to be added.

        """
        self.factory.parent_stack.append(node)


class XMLFactory(object):
    """Base class for XML factories.
    
    """
    def __init__(self, tree):
        self._orig_root = copy.deepcopy(tree.getroot())
        self._tree = None
        self.handlers = {}
        self.parent_stack = []
        self.stack = ParentStack(self)
    
    def _handler_for_node(self, node):
        """Returns handler for given element type.

        :param node: node for which handler should be returned.
        :returns: :class:`ElementHandler`.

        """
        try:
            node_handling = self.handlers[clean_tag_name(node)]
        except KeyError:
            # NOTE: Fallback to default handler if available.
            try:
                node_handling = self.handlers['*']
            except KeyError:
                raise HandlerMissing('No handler defined for element "%s"' %
                                        clean_tag_name(node))
        if isinstance(node_handling, tuple):
            handler, generator = node_handling
        else:
            handler = node_handling
        return handler

    def _generator_for_node(self, node):
        """Returns generator for given element type.

        :param node: node for which generator should be returned.
        :returns: :class:`DataGenerator` or :class:`NoneType`.

        """
        generator = None
        try:
            node_handling = self.handlers[clean_tag_name(node)]
        except KeyError:
            # NOTE: Fallback to default handler if available.
            try:
                node_handling = self.handlers['*']
            except KeyError:
                raise HandlerMissing('No handler defined for element "%s"' %
                                        clean_tag_name(node))
        if isinstance(node_handling, tuple):
            handler, generator = node_handling
        return generator

    def is_node_empty(self, node):
        """Moves down the tree in order to find any node that after processing
        returns result that is not :class:`NoneType`. Used for searching for
        possible options in :class:`RandomSelectionHandler`.

        :param node: root node for tree traversal.
        :returns: Returns :class:`etree.Element`, :class:`unicode` or
            :class:`tuple`\(:class:`etree.Element`, :class:`etree.Element`).

        """
        result = self.node_process(node)
        child_result = None
        if result is None:
            for child in node.getchildren():
                self.is_node_empty(child)
        else:
            if isinstance(result, tuple):
                new_node, next_node = result
                for child in next_node.getchildren():
                    child_result = self.is_node_empty(child)
                    if child_result is not None:
                        return child_result
            elif isinstance(result, etree._Element):
                for child in node.getchildren():
                    child_result = self.is_node_empty(child)
                    if child_result is not None:
                        return child_result
            elif isinstance(result, (str, unicode, int)):
                return unicode(node)
            else:
                # TODO: Type not supported.
                raise Exception
        return not bool(child_result)

    def _traverse_tree(self, node):
        """Moves down the tree processing each node.

        :param node: root node for tree traversal.

        """
        result = self.node_process(node)
        if result is None:
            for child in node.getchildren():
                self._traverse_tree(child)
        else:
            if isinstance(result, tuple):
                new_node, next_node = result
                if new_node is None:
                    for child in next_node.getchildren():
                        self._traverse_tree(child)
                else:
                    parent = self.stack.last()
                    parent.append(new_node)
                    if next_node.getchildren() is not None:
                        self.stack.add(new_node)
                    for child in next_node.getchildren():
                        self._traverse_tree(child)
                    self.stack.pop()
            elif isinstance(result, etree._Element):
                # NOTE: Redundant, keeps thinking pattern though. To be changed
                #       at some point.
                new_node = result
                parent = self.stack.last()
                if parent is None:
                    self._tree = etree.ElementTree()
                    self._tree._setroot(new_node)
                else:
                    parent.append(new_node)
                if node.getchildren() is not None:
                    self.stack.add(new_node)
                for child in node.getchildren():
                    self._traverse_tree(child)
                generator = self._generator_for_node(node)
                if generator and not new_node.getchildren():
                    data = generator.generate(node)
                    if data:
                        if not new_node.text:
                            new_node.text = unicode(data)
                self.stack.pop()
            # TODO: Unicode not enough?
            elif isinstance(result, (str, unicode, int)):
                new_value = result
                self.stack.last().text = unicode(new_value)
            else:
                # TODO: Type not supported.
                raise Exception

    def node_process(self, node):
        """Processes node with a handler assigned by
        :meth:`register_default_handlers` in :class:`XMLFactory` and descendants.

        :param node: node to be processed.
        :returns: Depending on the handler, returns :class:`NoneType`,
            :class:`tuple`\(:class:`etree.Element`, :class:`etree.Element`),
            :class:`etree.Element` or :class:`unicode`.

        """
        handler = self._handler_for_node(node)
        result_node = handler.process(node)
        return result_node

    def register_default_handlers(self):
        """Called each time a new factory is instantinated. May be overridden
        by subclasses' `__init__()` methods.
        
        **Example**

        .. code-block:: python

            def register_default_handlers(self):
                self.register_handler('*', CopyHandler)

        """
        pass

    def register_handler(self, element_name, handling_args):
        """Registers handler for given tag name.

        :param element_name: string containing name of the handled element.
        :param handling_args: tuple or :class:`ElementHandler`. If tuple passed,
            the second argument must be a :class:`DataGenerator` class.

        """
        if isinstance(handling_args, tuple):
            self.handlers.update({
                element_name: (handling_args[0](self), handling_args[1](self))
            })
        elif issubclass(handling_args, ElementHandler):
            self.handlers.update({element_name: handling_args(self)})
        else:
            raise ImproperlyConfigured('Invalid configuration parameters for'
                            'register_handler()')

    def flush_handlers(self):
        """Resets handlers.

        """
        self.handlers = {}

    def run(self):
        """Main entry point for :class:`XMLFactory`.

        :returns: :class:`etree.Tree`.

        """
        self._traverse_tree(self._orig_root)
        return self._tree 

    def rewrite_with_defaults(self):
        """Default conversion method for :class:`XMLFactory`.

        """
        if not self.handlers:
            self.register_default_handlers()
        return self.run()
