from xmltrans.factories import XMLFactory
from xmltrans.handlers import (
    IgnoreHandler,
    RandomSelectionHandler,
)

from example.handlers import (
    RandomValueHandler,
    TranscriptionHandler,
)
from example.generators import (
    ValidDataGenerator,
    StringDataGenerator,
    MixedDataGenerator,
)


class XMLFromSchema(XMLFactory):
    """Factory for generating random XML examples from Schema.

    **Example**

    .. code-block:: python

        >>> from lxml import etree
        >>> from example.factories import XMLFromSchema
        >>> tree = etree.parse('example.xsd')
        >>> xml_factory = XMLFromSchema(tree)
        >>> new_tree = xml_factory.valid_xml_and_data()
        >>> print etree.tostring(new_tree.getroot(), pretty_print=True) 

    """
    def register_default_handlers(self):
        """Default handlers when no other specified, used with :meth:`rewrite`.

        """
        super(SchemaConversion, self).register_default_handlers()
        self.register_handler('annotation', IgnoreHandler)
        self.register_handler('documentation', IgnoreHandler)
        self.register_handler('schema', IgnoreHandler)
        self.register_handler('complexType', IgnoreHandler)
        self.register_handler('simpleType', IgnoreHandler)
        self.register_handler('enumeration', IgnoreHandler)
        self.register_handler('sequence', IgnoreHandler)
        self.register_handler('choice', RandomSelectionHandler)
        self.register_handler('restriction', RandomValueHandler)
        self.register_handler('element', TranscriptionHandler)

    def valid_xml_and_data(self):
        """Converts XML Schema to a XML document with valid structure and data.

        """
        self.flush_handlers()
        self.register_handler('annotation', IgnoreHandler)
        self.register_handler('documentation', IgnoreHandler)
        self.register_handler('schema', IgnoreHandler)
        self.register_handler('complexType', IgnoreHandler)
        self.register_handler('simpleType', IgnoreHandler)
        self.register_handler('enumeration', IgnoreHandler)
        self.register_handler('sequence', IgnoreHandler)
        self.register_handler('choice', RandomSelectionHandler)
        self.register_handler('restriction', RandomValueHandler)
        self.register_handler('element', (TranscriptionHandler, ValidDataGenerator))
        return self.run()

    def valid_xml_forced_strings(self):
        """Converts XML Schema to a XML document with valid structure and
        random string values assigned to each empty element.

        """
        self.flush_handlers()
        self.register_handler('annotation', IgnoreHandler)
        self.register_handler('documentation', IgnoreHandler)
        self.register_handler('schema', IgnoreHandler)
        self.register_handler('complexType', IgnoreHandler)
        self.register_handler('simpleType', IgnoreHandler)
        self.register_handler('enumeration', IgnoreHandler)
        self.register_handler('sequence', IgnoreHandler)
        self.register_handler('choice', RandomSelectionHandler)
        self.register_handler('restriction', RandomValueHandler)
        self.register_handler('element', (TranscriptionHandler, StringDataGenerator))
        return self.run() 

    def valid_xml_mixed_data(self):
        """Converts XML Schema to a XML document with valid structure and
        random string values assigned only to elements for which type is not
        specified or not supported.

        """
        self.flush_handlers()
        self.register_handler('annotation', IgnoreHandler)
        self.register_handler('documentation', IgnoreHandler)
        self.register_handler('schema', IgnoreHandler)
        self.register_handler('complexType', IgnoreHandler)
        self.register_handler('simpleType', IgnoreHandler)
        self.register_handler('enumeration', IgnoreHandler)
        self.register_handler('sequence', IgnoreHandler)
        self.register_handler('choice', RandomSelectionHandler)
        self.register_handler('restriction', RandomValueHandler)
        self.register_handler('element', (TranscriptionHandler, MixedDataGenerator))
        return self.run() 