from flask import jsonify, request, Flask, make_response
import logging
import re
from multiprocessing import Process
import requests
from requests.exceptions import RequestException
import json
import socket
import time
from collections import OrderedDict

mock_server = Flask(__name__)
_rules = OrderedDict()
_requests = []


# @mock_server.route("/test", methods=['GET'])
# def test():
#     return "OK"

@mock_server.route("/mock/shutdown", methods=['GET'])
def shutdown():
    func = request.environ.get('werkzeug.server.shutdown')
    if func is None:
        raise RuntimeError('Not running with the Werkzeug Server')
    func()
    return 'Server shutting down...'


@mock_server.route("/mock/responses", methods=['POST'])
def add_response():
    """
    This method adds new responses to the mock.
    To add a response send a POST request with a payload like this:

    {
        "url_filter": ".*",
        "headers": {
            "Accept": "text/xml"
        },
        "body": "Sample body",
        "status_code": 200
    }

    Server will validate each matching rule and apply the first match
    If there is no match, it will return a 500 response
    """
    try:
        payload = request.get_json(force=True)
        assert isinstance(payload, dict)
    except:
        logging.error("Payload is not a valid JSON string")
        logging.error(request.data)
        return "Payload is not valid JSON string", 400

    #Parse data from request
    if "url_filter" in payload.keys():
        key = payload["url_filter"]
        try:
            url_filter = re.compile(payload["url_filter"])
        except Exception as e:
            logging.error("url_filter is not a valid regular expression")
            logging.error(payload["url_filter"])
            return "url_filter is not a valid regular expression:\\n%s" % e.message, 400
    else:
        url_filter = re.compile('.*')
        key = '.*'

    if "headers" in payload.keys():
        if type(payload["headers"]) is dict:
            headers = payload["headers"]
        else:
            return "headers is not a dictionary:\\n%s" % payload["headers"], 400
    else:
        headers = {}

    if "body" in payload.keys():
        body = payload["body"]
    else:
        body = ""

    if "status_code" in payload.keys():
        status_code = payload["status_code"]
    else:
        status_code = 200

    #Save parsed data
    new_rule = {"url_filter": url_filter, "headers": headers, "body": body, "status_code": status_code}
    _rules[key] = new_rule

    return "OK"


@mock_server.route("/mock/responses", methods=['DELETE'])
def clear_responses():
    """
    Delete existing responses
    """
    _rules.clear()
    return "All rules were deleted"


@mock_server.route("/mock/responses", methods=['GET'])
def get_responses():
    """
    Get all responses
    """
    rules_as_text = []
    for rule in _rules.values():
        #make a copy so we don't modify original
        rule = rule.copy()

        #Convert regex to str
        rule["url_filter"] = rule["url_filter"].pattern

        #Add rule to list
        rules_as_text.append(rule)

    return jsonify(rules=rules_as_text)


@mock_server.route("/mock/requests", methods=['DELETE'])
def clear_requests():
    """
    Delete existing requests
    """
    del _requests[0:len(_requests)]
    return "All requests were deleted"


@mock_server.route("/mock/requests", methods=['GET'])
def get_requests():
    """
    Get all requests
    """
    return jsonify(requests=_requests)

@mock_server.route('/', defaults={'path': ''}, methods=['GET', 'POST', 'PUT', 'DELETE'])
@mock_server.route('/<path:path>', methods=['GET', 'POST', 'PUT', 'DELETE'])
def catch_all(path):
    """
    This method will catch all requests for which there are no explicit routes.
    Here is where we build responses based on the rules that have been configured
    It will go though the list of rules and apply one by one until a match is found.
    If there is no match, it will return a 500 response
    """
    _requests.append(request.url)
    for rule in _rules.values():
        regex = rule["url_filter"]
        if regex.search("/" + path):
            response = make_response()
            response .headers = rule["headers"]
            response .data = rule["body"]
            response .status_code = rule["status_code"]
            return response

    # Default values returned when there wasn't a match in the rules
    return "Mock has not been configured", 500


class ApiMockServer(Process):
    """
    Helper Class to interact with the ApiMockServer
    Provides methods to add, remove mock objects as well as incoming requests
    """

    server = mock_server

    def __init__(self, port):
        Process.__init__(self)
        self.port = port
        self.host = "0.0.0.0"  # host IP to be server by the Flask server
        self.localhost = self.host_ip()  #  IP to communicate with server
        self.base_url = "http://{host}:{port}".format(host=self.localhost, port=self.port)
        self.responses_url = self.base_url + "/mock/responses"
        self.requests_url = self.base_url + "/mock/requests"
        self.shutdown_url = self.base_url + "/mock/shutdown"

    def run(self):
        """
        start the mock server
        """
        self.server.run(host=self.host, port=self.port)

    def stop(self):
        """
        Shutdown the server and terminate process
        :return:
        """
        self.wait_for_server_to_start()
        requests.get(self.shutdown_url)
        self.terminate()

    def add_response(self, url_filter=".*", status_code=200, headers=None, body="", data=None):
        """
        Add the response to the mock server
        if the payload is provided, other parameters are ignored

        :param url_filter: Regular expression to match the url path
        :param status_code: Expected status code
        :param headers: Headers as a dictionary
        :param body: response body as plain text
        :param payload: a python dictionary
        :return: HTTP status code or error message if any error occurred
        """
        # if payload is provided, all other other parameters woll be ignored
        self.wait_for_server_to_start()
        if data is None:
            if not headers:
                headers = {}
            data = {"url_filter": url_filter,
                       "status_code": status_code,
                       "headers": headers,
                       "body": body}
        try:
            response = requests.post(self.responses_url, data=json.dumps(data))
            return response
        except RequestException as ex:
            return ex.message

    def get_responses(self):
        """
        Returns all the responses stored on the MockServer
        :return:
        """
        self.wait_for_server_to_start()
        response = requests.get(self.responses_url)
        if response:
            return response
        return None

    def clear_responses(self):
        """
        Delete all the responses stored on the MockServer
        :return:
        """
        self.wait_for_server_to_start()
        return requests.delete(self.responses_url)

    def get_requests(self):
        """
        Returns all the requests stored on the MockServer
        :return:
        """
        self.wait_for_server_to_start()
        response = requests.get(self.requests_url)
        if response:
            return response
        return None

    def clear_requests(self):
        """
        Delete all the requests stored on the MockServer
        :return:
        """
        self.wait_for_server_to_start()
        return requests.delete(self.requests_url)

    def host_ip(self):
        """
        Dirty hack to return the localhost IP,
        works only when you have an internet connection
        :return:
        """
        return [(s.connect(('8.8.8.8', 80)), s.getsockname()[0], s.close()) for s in [socket.socket(socket.AF_INET, socket.SOCK_DGRAM)]][0][1]

    def _is_server_running(self):
        """
        check if the server is running and port is open
        """
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        result = sock.connect_ex((self.host_ip(),self.port))
        return result == 0

    def wait_for_server_to_start(self):
        """
        Latency to avoid the access when the server is not started yet
        """
        while not self._is_server_running():
            time.sleep(0.1)

    def get_base_url(self):
        return self.base_url

if __name__ == '__main__':
    #to us as standalone application
    #run 'python ApiMockServer.py' 
    api_server = ApiMockServer(port=10000)
    api_server.start()
