import os
from setuptools import setup, find_packages
from distutils.command.install import install as _install
import sys


def get_version_filename():
    """
    Return full filename where package version number is stored
    """
    parent_dir = os.path.dirname(__file__)
    return os.path.abspath(os.path.join(parent_dir, "VERSION.txt"))


def get_version():
    """
    Returns package's version number.\n
    Version is read from file VERSION.txt (or file returned by get_version_filename())
    """
    try:
        f = open(get_version_filename(), 'r')
        try:
            #Get version from file, removing carriage return
            version = f.read().strip()
            return version
        finally:
            f.close()
    except IOError:
        sys.stderr.write("File %s doesn't exist. Can't get package version" % get_version_filename())
        exit(-1)


#Get requirements from requirements.txt file
def get_requirements():
    with open('requirements.txt') as f:
        install_requires = f.readlines()

    #Remove "\n" from each item
    requirements = map(lambda s: s.strip(), install_requires)
    return requirements

def get_assets(folders):
    """
    Returns all files under the folders specified.
    This will reduce maintenance of this script in case there are new files
     added under asset
    @type folders: list[str]
    @param folders: List of folders to search. Folders are relative to TLib\tlib folder
    """
    parent_dir = os.path.dirname(__file__)

    #Get TLib and asset folders
    tlib_dir = os.path.abspath(os.path.join(parent_dir, "tlib"))

    #This will contain the list of files
    matches = []

    for folder in folders:
        search_dir = os.path.abspath(os.path.join(tlib_dir, folder))

        for root, dirnames, filenames in os.walk(search_dir):
            files = [os.path.relpath(os.path.join(root, filename), tlib_dir) for filename in filenames if not filename.endswith(".pyc")]
            # files = [file for file in all_files is not file.endswith(".pyc")]
            matches.extend(files)

    return matches


setup(
    name='tlib',
    version=get_version(),
    packages=find_packages(),
    author='Quality Assurance Automation @ Yellow Pages Group',
    author_email="none@dummy.com",                                      # To avoid warnings
    url='http://172.27.13.48:8081',
    description="""Packages to help the wayward QA tester.
        Contains :
            - HttpRequestor : Facilitates communication to REST APIs
            - SoapRequestor : Facilitates communication to SOAP APIs
            - SpiraModerator : Fetches and updates data from SpiraTeam
            - JiraModerator : Fetches and updates issues in Jira
            - WtpHelper : Sends test requests to WebPagetest and returns test results
            """,
    install_requires=get_requirements(),
    # the following makes a plugin available to py.test
    entry_points={
        'pytest11': [
            'selenium = pytest_auto.selenium_plugin',
            'yp_auto = pytest_auto.yp_auto',
        ]
    },

    package_data={'tlib': get_assets(["asset", "config", "templates"]) },
)

