"""
Pytest plugin with commonly used fixtures
"""
import os
import pytest
from ConfigParser import ConfigParser, NoSectionError
from tlib.base import LogHelper
from tlib.base import TLibHelper


# noinspection PyShadowingNames
@pytest.fixture(scope='class')
def tlib_logger(request):
    """
    Returns logger with name tlib.\n
    This logger should only be used by TLib modules. To log from a test or test module use test_logger\n
    Logger will get flushed at the end to ensure data is not lost if tests exit abnormally
    """
    return LogHelper.get_tlib_logger(request)


# noinspection PyShadowingNames
@pytest.fixture(scope='class')
def adb_logger(request):
    """
    Returns logger with name adb\n
    This logger should only be used by TLib modules interacting with ADB.\n
    To log from a test or test module use test_logger\n
    Logger will get flushed at the end to ensure data is not lost if tests exit abnormally
    """
    return LogHelper.get_adb_logger(request)


# noinspection PyShadowingNames
@pytest.fixture(scope='class')
def test_logger(request):
    """
    Returns logger that can be used to log test information\n
    Logger will get flushed at the end to ensure data is not lost if tests exit abnormally
    """
    return LogHelper.get_test_logger(request)


def pytest_addoption(parser):
    """
    Options supported by pytest selenium plug-in
    """
    group = parser.getgroup("tlib", "General command line options provided by TLib")

    group.addoption(
        "--environment",
        action="store",
        dest="environment",
        help="environment used to run tests")

    group.addoption(
        "--base_url",
        action="store",
        dest="base_url",
        help="base url of the server used for testing. This can be used for web services as well as web sites")

    group.addoption(
        "--apk_path",
        action="store",
        dest="apk_path",
        default=None,
        help="mobile app path of the mobile native app used for testing")


#noinspection PyUnresolvedReferences
#noinspection PyShadowingNames
@pytest.fixture(scope="class")
def base_url(request, tlib_logger):
    """
    Returns value of --base_url command line parameter
    """
    return TLibHelper.get_base_url(request, tlib_logger)


#noinspection PyUnresolvedReferences
#noinspection PyShadowingNames
@pytest.fixture(scope="class")
def environment_config(request, tlib_logger):
    """
    Returns configuration for the environment specified in the --environment command line parameter
    This function will try to find a file config\config.ini on the current folder or on the parents,
    up to two levels up.

    This fixture returns a dict like this:
    {
        'mysql_host': 'db.ci.qa.ypg.com',
        'mysql_user': 'ci',
        'mysql_passwd': '21345'
    }

    @rtype dict
    """
    environment = request.config.getoption("environment")

    if environment is None:
        #noinspection PyUnresolvedReferences
        tlib_logger.error("Parameter --environment was not specified, test can't run")
        raise RuntimeError("Parameter --environment was not specified, test can't run")
        return None
    else:
        #Go up to three levels to find config folder. Search starts in current directory
        curr_folder = os.path.dirname(request.fspath.strpath)
        config_file = None
        for i in range(1, 4):
            #Go up one level
            curr_folder = os.path.abspath(os.path.join(curr_folder, os.pardir))

            #Check if there is a 'config/config.ini' file
            f = os.path.join(curr_folder, 'config', 'config.ini')
            if os.path.exists(f):
                config_file = f
                break

        if config_file is None:
            raise IOError("Couldn't find folder 'logs'. Create it under your project folder and try again")

        prj_config = ConfigParser()
        prj_config.read(config_file)

        try:
            cfg = prj_config.items("env_" + environment)
            return dict(cfg)
        except NoSectionError:
            tlib_logger.error("There is no section 'env_%s' in config file" % environment)
            raise RuntimeError("There is no section 'env_%s' in config file" % environment)

