import pytest
import os
from selenium import webdriver as selenium_webdriver
from tlib.base import AndroidHelper
from tlib.base.TestHelper import webdriver_chrome_executable, webdriver_ie_executable
from _pytest.python import FixtureRequest
import logging
from appium.webdriver.webdriver import WebDriver as appium_webdriver
from selenium.common.exceptions import WebDriverException

def pytest_addoption(parser):
    """
    Options supported by pytest's selenium plug-in
    """
    group = parser.getgroup("selenium", "Testing with the selenium framework")

    group.addoption(
        "--mobile_base_url",
        action="store",
        dest="mobile_base_url",
        help="mobile base url of the mobile web server used for testing")

    group.addoption(
        "--browser",
        action="store",
        dest="browser",
        choices=["chrome", "firefox", "firefox_no_js", "ie", "android", "chrome_android"],
        help="Browser used for running test cases")

    group.addoption(
        "--chromedriver_version",
        action="store",
        dest="chromedriver_version",
        default="latest",
        help="Version of chromedriver to use")

    group.addoption(
        "--serial_id",
        action="store",
        dest="serial_id",
        help="Serial of the android device or simulator where Android tests will be run")


#noinspection PyShadowingNames,PyUnresolvedReferences
@pytest.fixture(scope='class')
def driver(request, tlib_logger, adb_logger):
    """
    Fixture to creates a Webdriver object to interact with native apps specified by command line parameters
    @param request: Object that gives access to the requesting test context
    @type request: FixtureRequest
    @param tlib_logger: logger for tlib related actions
    @type tlib_logger: logging.Logger
    """
    serial_id = request.config.getoption("serial_id")

    apk_path = request.config.getoption("apk_path")
    if apk_path is None:
        tlib_logger.error("parameter --apk_path was not specified")
        raise RuntimeError("parameter --apk_path was not specified")

    if not os.path.exists(apk_path):
        tlib_logger.error("Selendroid coulnd't be found under '%s'" % apk_path)
        raise RuntimeError("Selendroid coulnd't be found under '%s'" % apk_path)

    tlib_logger.debug("Creating selendroid driver")

    #Start selendroid server
    process = AndroidHelper.start_selendroid_server(tlib_logger, adb_logger, apk_path)
    if process:
        request.addfinalizer(process.kill)

    try:
        return selendroid_driver(tlib_logger, serial_id)
    except WebDriverException, e:
        raise RuntimeError("An error occurred while trying to look for device/emulator")


def selendroid_driver(tlib_logger, serial_id):
    """
    Function to create Webdriver object to interact with native apps
    @param tlib_logger: logger for tlib related actions
    @type tlib_logger: logging.Logger
    @param serial_id: Serial Id of android device. If not given default android device will be used
    @type serial_id: str
    """
    app_id = AndroidHelper.get_android_app_id()
    capabilities = {'aut': app_id}
    if serial_id is None:
        tlib_logger.warn("Parameter --serial_id was not given. Running test on default device")
    else:
        capabilities['serial'] = serial_id
    dr = selenium_webdriver.Remote(desired_capabilities=capabilities)

    return dr


#noinspection PyShadowingNames,PyUnresolvedReferences
@pytest.fixture(scope='class')
def browser(request, tlib_logger, adb_logger, chromedriver_version):
    """
    Fixture to create a browser as specified by the command line parameter --browser\n
    """
    browser = request.config.getoption("browser")
    return get_browser(request, tlib_logger, adb_logger, browser, chromedriver_version)


@pytest.fixture(scope='class')
def chromedriver_version(request, tlib_logger):
    """
    Fixture to create a browser as specified by the command line parameter --browser\n
    """
    version = request.config.getoption("chromedriver_version")
    tlib_logger.info("Using chromedriver version = '%s'" % version)
    return version


def get_browser(request, tlib_logger, adb_logger, browser_str, chromedriver="latest"):
    """
    Creates a Webdriver object to interact with browser specified in browser_str parameter
    @param request: Object that gives access to the requesting test context
    @type request: FixtureRequest
    @param tlib_logger: logger for tlib related actions
    @type tlib_logger: logging.Logger
    @param adb_logger: logger for adb related actions
    @type adb_logger: logging.Logger
    @param browser_str: browser to launch. It cna be firefox, chrome, ie or android
    @type browser_str: str
    @param chromedriver: Version of ChromeDriver to use. Pass 'latest' to use the latest version
    @type chromedriver: str
    """
    b = None
    process = None

    #Don't create a browser object is --browser was not specified
    if browser_str is None:
        tlib_logger.error("Parameter browser was not specified")
        raise RuntimeError("Parameter --browser was not specified")
    else:
        if browser_str == "ie":
            tlib_logger.debug("Creating webdriver for IE")
            b = selenium_webdriver.Ie(webdriver_ie_executable())
            b.maximize_window()
        elif browser_str == "chrome":
            tlib_logger.debug("Creating webdriver for Chrome")
            b = selenium_webdriver.Chrome(webdriver_chrome_executable(chromedriver))
            b.maximize_window()
        elif browser_str == "firefox":
            tlib_logger.debug("Creating webdriver for Firefox")
            b = selenium_webdriver.Firefox()
            b.maximize_window()
        elif browser_str == "firefox_no_js":
            tlib_logger.debug("Creating webdriver for Firefox with JavaScript disabled")
            profile = selenium_webdriver.FirefoxProfile()
            profile.set_preference("javascript.enabled", False)
            b = selenium_webdriver.Firefox(profile)
            b.maximize_window()
        elif browser_str == "android":
            tlib_logger.debug("Creating webdriver for Android")

            serial_id = request.config.getoption("serial_id")
            if serial_id is None:
                tlib_logger.warn("Parameter --serial_id was not given. Running test on default device")

            #Start android server
            AndroidHelper.setup_webdriver(tlib_logger, adb_logger, serial_id)
            b = selenium_webdriver.Remote(command_executor='http://localhost:8080/wd/hub',
                                          desired_capabilities=selenium_webdriver.DesiredCapabilities.ANDROID)
        elif browser_str == "chrome_android":
            tlib_logger.debug("Running Chrome on android")

            desired_caps = {}

            serial_id = request.config.getoption("serial_id")
            if serial_id is not None:
                desired_caps['udid'] = serial_id
            else:
                desired_caps['defaultDevice'] = True

            desired_caps['deviceName']=''
            desired_caps['platformName'] = 'Android'
            desired_caps['browserName'] = 'Chrome'
            desired_caps['noReset'] = True


            #Start selendroid server
            process = AndroidHelper.start_appium_server(tlib_logger, adb_logger)
            b = appium_webdriver(desired_capabilities=desired_caps)
        else:
            #noinspection PyUnresolvedReferences
            raise RuntimeError("{browser}s is not a valid browser".format(browser=browser_str))

        def fin():
            # noinspection PyStatementEffect
            b.quit()
            if browser_str == "android":
                AndroidHelper.teardown_webdriver(tlib_logger, adb_logger, serial_id)

            if process:
                process.kill

        request.addfinalizer(fin)

    # set browser name
    b.browser_name = browser_str
    return b


#noinspection PyUnresolvedReferences
@pytest.fixture(scope="class")
def mobile_base_url(request, tlib_logger):
    """
    Returns value of --mobile_base_url command line parameter
    """
    if request.config.getoption("mobile_base_url") is None:
        #noinspection PyUnresolvedReferences
        tlib_logger.error("parameter --mobile_base_url was not specified")
        raise RuntimeError("parameter --mobile_base_url was not specified")
        return None
    else:
        return request.config.getoption("mobile_base_url")
