import BaseHTTPServer
from SimpleHTTPServer import SimpleHTTPRequestHandler
import socket
import json
from SocketServer import ThreadingMixIn
import struct
import os


class _MyHandler(SimpleHTTPRequestHandler):
    """
    Class for handling POST and GET request to our mock server.  To be used with the ApiMockServer
    """

    def __init__(self, req, client_addr, server):
        SimpleHTTPRequestHandler.__init__(self, req, client_addr, server)

    def do_GET(self):
        """
        Handles the GET requests for retrieving the mock data that was last stored in memory

        Returns : (json) 'warning' = "No data in memory" if nothing has been previously stored.  Returns a mock json if
        something was previously stored.
        """
        self.send_response(200)
        self.send_header("Content-type", "text/json")
        self.end_headers()
        output = {}

        if not self.server.jsonData or 'warning' in self.server.jsonData:
            output["warning"] = "No data in memory"
        else:
            output = self.server.jsonData

        self.wfile.write(json.dumps(output))

    #noinspection PyPep8Naming
    def do_POST(self):
        """
        Handles the POST request for putting a mock json response into memory for later retrieval with GET

        Returns : (json) 'data' = 'Data has been saved to memory' if the data sent successfully saved.
        """
        try:
            self.send_response(200)
            self.send_header("Content-type", "text/json")
            length = int(self.headers.getheader('content-length'))
            data = self.rfile.read(length)
            self.server.jsonData = json.loads(data)
            self.end_headers()
            self.wfile.write('Data has been saved to memory')
        except ValueError:
            self.server.logger.error("The data sent to ApiMockServer was not in json format.  nothing saved")
            self.send_error(400, 'Not a json object. Data has not been saved to memory')


class ApiMockServer(ThreadingMixIn, BaseHTTPServer.HTTPServer):

    server_address = ('0.0.0.0', 8000)
    json_data = {}
    logger = None
    my_ip = None

    def __init__(self, logger):

        BaseHTTPServer.HTTPServer.__init__(self, self.server_address, _MyHandler)
        self.logger = logger
        self.my_ip = self.get_lan_ip()

    @staticmethod
    def get_interface_ip(ifname):
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        #noinspection PyUnresolvedReferences
        return socket.inet_ntoa(fcntl.ioctl(
                                s.fileno(),
                                0x8915, struct.pack('256s', ifname[:15])
                                )[20:24])

    def get_lan_ip(self):
        ip = socket.gethostbyname(socket.gethostname())
        if ip.startswith("127.") and os.name != "nt":
            interfaces = ["eth0", "eth1", "eth2", "wlan0", "wlan1", "wifi0", "ath0", "ath1", "ppp0"]

            for ifname in interfaces:
                try:
                    ip = self.get_interface_ip(ifname)
                    break

                except IOError:
                    pass
        return ip



