import os
import socket
import subprocess
import pytest
import logging
import tempfile

def tlib_folder():
    """
    Returns tlib's absolute path
    """
    file_folder = os.path.dirname(__file__)
    return os.path.abspath(os.path.join(file_folder, os.pardir))


def tlib_asset_folder():
    """
    Returns absolute path of tlib's asset folder
    """
    return os.path.abspath(os.path.join(tlib_folder(), "asset"))


def tlib_config_folder():
    """
    Returns absolute path of tlib's config folder 
    """
    return os.path.abspath(os.path.join(tlib_folder(), "config"))


def tlib_template_folder():
    """
    Returns absolute path of tlib's template folder\n
    Template folder contains jinja templates used for generation of reports\n
    like screenshots
    """
    return os.path.abspath(os.path.join(tlib_folder(), "templates"))


def webdriver_chrome_executable():
    """
    Returns path to Chrome WebDriver executable
    """
    return os.path.join(tlib_asset_folder(), "selenium", "chromedriver.exe")


def webdriver_ie_executable():
    """
    Alias for webdriver_ie32_executable
    """
    return webdriver_ie32_executable();


def webdriver_ie32_executable():
    """
    Returns path to IE WebDriver executable
    """
    return os.path.join(tlib_asset_folder(), "selenium", "IE", "Win32_2.39.0", "IEDriverServer.exe")


def webdriver_ie64_executable():
    """
    Returns path to IE WebDriver executable
    """
    return os.path.join(tlib_asset_folder(), "selenium", "IE", "x64_2.39.0", "IEDriverServer.exe")


def android_server_apk():
    """
    Returns path to Webdriver for android
    """
    return os.path.join(tlib_asset_folder(), "selenium", "android-server-2.21.0.apk")


def is_valid_ip(ip):
    """
    Returns true if IP parameter is a valid IP address.\n
    Currently IPs in this format are valid:\n
    X.X.X.X\n
    X.X.X.X:PORT

    @type ip: str
    @return: bool
    """

    if ip is None:
        return False

    try:
        (ip, port) = ip.split(":", 1)
    except ValueError:
        #There is no ':' in the string
        port = None

    try:
        socket.inet_aton(ip)
    except socket.error:
        return False

    if port is not None:
        try:
            return (int(port) >= 1) and (int(port) <= 65535)
        except ValueError:
            #Not a valid integer
            return False

    return True

def run_command(logger, command, fail_on_error=True):
    """
    Run a command and skip test if exit code is not 0

    Example:
    Run 'adb devices' and don't skip test if commands returns non-zero status
    run_command(logger, ["adb", "devices"], fail_on_error=False)

    @param logger: logger for debugging purposes
    @type logger: logging.logger
    @param command: Command to run
    @type command: list
    @param fail_on_error: When True, skip test if command returned a non-zero exit code
    @type fail_on_error: bool
    @rtype: list
    @return: Returns a list with stdout and stderr output
    """

    fd_out = tempfile.NamedTemporaryFile(delete=True)
    #With delete = True option, the files will be automatically removed after close
    fd_err = tempfile.NamedTemporaryFile(delete=True)
    try:
        process = subprocess.Popen(command, stdout=fd_out, stderr=fd_err)
    except WindowsError as e:
        logger.error(r"Problem running command, skipping test.\n%s" % e)
        # noinspection PyUnresolvedReferences
        if fail_on_error:
            pytest.fail(r"Problem running command, skipping test.\n%s" % e)

    # noinspection PyUnboundLocalVariable
    process.communicate()
    fd_out.seek(0)
    fd_err.seek(0)
    out = (fd_out.read(), fd_err.read())
    fd_out.close()
    fd_err.close()
    errcode = process.returncode

    if (errcode != 0) and fail_on_error:
        logger.error(r"Program  exited with code {errcode}, skipping test\n{out}".format(errcode=errcode, out=out))
        # noinspection PyUnresolvedReferences
        pytest.fail(r"Program exited with code {errcode}, skipping test\n{out}".format(errcode=errcode, out=out))

    return out

