import pytest
import logging
from selenium import webdriver
from tlib.base import AndroidHelper
from tlib.base import TestHelper
from tlib.base.TestHelper import webdriver_chrome_executable, webdriver_ie_executable
from _pytest.python import FixtureRequest


def pytest_addoption(parser):
    """
    Options supported by pytest's selenium plug-in
    """
    group = parser.getgroup("selenium", "Testing with the selenium framework")
    group.addoption(
        "--base_url",
        action="store",
        dest="base_url",
        help="base url of the server used for testing")

    group.addoption(
        "--mobile_base_url",
        action="store",
        dest="mobile_base_url",
        help="mobile base url of the mobile web server used for testing")

    group.addoption(
        "--browser",
        action="store",
        dest="browser",
        choices=["chrome", "firefox", "ie", "android"],
        help="Browser used for running test cases.")

    group.addoption(
        "--serial_id",
        action="store",
        dest="serial_id",
        help="Serial of the android device or simulator where Android tests will be run")


#noinspection PyShadowingNames,PyUnresolvedReferences
@pytest.fixture(scope='session')
def browser(request, tlib_logger, adb_logger):
    """
    Fixture to create a browser as specified by the command line parameter --browser\n
    """
    browser = request.config.getoption("browser")
    return get_browser(request, tlib_logger, adb_logger, browser)


def get_browser(request, tlib_logger, adb_logger, browser_str):
    """
    Creates a Webdriver object to interact with browser specified in browser_str parameter
    @param request: Object that gives access to the requesting test context
    @type request: FixtureRequest
    @param tlib_logger: logger for tlib related actions
    @type tlib_logger: logging.logger
    @param adb_logger: logger for adb related actions
    @type adb_logger: logging.logger
    @param browser_str: browser to launch. It cna be firefox, chrome, ie or android
    @type browser_str: str
    """
    global b

    #Don't create a browser object is --browser was not specified
    if browser_str is None:
        tlib_logger.warn("Parameter browser was not specified")
        pytest.fail("Parameter browser was not specified")
    else:
        if browser_str == "ie":
            tlib_logger.debug("Creating webdriver for IE")
            b = webdriver.Ie(webdriver_ie_executable())
            b.maximize_window()
            request.addfinalizer(b.quit)
        elif browser_str == "chrome":
            tlib_logger.debug("Creating webdriver for Chrome")
            b = webdriver.Chrome(webdriver_chrome_executable())
            b.maximize_window()
            request.addfinalizer(b.quit)
        elif browser_str == "firefox":
            tlib_logger.debug("Creating webdriver for Firefox")
            b = webdriver.Firefox()
            b.maximize_window()
            request.addfinalizer(b.quit)
        elif browser_str == "android":
            tlib_logger.debug("Creating webdriver for Android")

            serial_id = request.config.getoption("serial_id")
            if serial_id is None:
                tlib_logger.warn("Parameter --serial_id was not given. Running test on default device")

            #Start android server
            AndroidHelper.setup_webdriver(tlib_logger, adb_logger, serial_id)
            b = webdriver.Remote(command_executor='http://localhost:8080/wd/hub',
                                 desired_capabilities=webdriver.DesiredCapabilities.ANDROID)

            def fin():
                b.quit
                AndroidHelper.teardown_webdriver(tlib_logger, adb_logger, serial_id)

            request.addfinalizer(fin)

        else:
            #noinspection PyUnresolvedReferences
            tlib_logger.warn("{browser}s is not a valid browser".format(browser=browser_str))
            pytest.fail("{browser}s is not a valid browser".format(browser=browser_str))

    return b


#noinspection PyUnresolvedReferences
@pytest.fixture(scope="session")
def base_url(request, tlib_logger):
    """
    Returns value of --base_url command line parameter
    """
    if request.config.getoption("base_url") is None:
        #noinspection PyUnresolvedReferences
        tlib_logger.warn("parameter --base_url was not specified")
        pytest.fail("parameter --base_url was not specified")
        return None
    else:
        return request.config.getoption("base_url")


#noinspection PyUnresolvedReferences
@pytest.fixture(scope="session")
def mobile_base_url(request, tlib_logger):
    """
    Returns value of --mobile_base_url command line parameter
    """
    if request.config.getoption("mobile_base_url") is None:
        #noinspection PyUnresolvedReferences
        tlib_logger.warn("parameter --mobile_base_url was not specified")
        pytest.fail("parameter --mobile_base_url was not specified")
        return None
    else:
        return request.config.getoption("mobile_base_url")
