import uuid
import subprocess
import os
import shutil
import struct
import re
from datetime import timedelta, datetime

import rarfile

from hachoir_metadata import extractMetadata
from hachoir_parser import createParser

from zope.interface import implements

from twisted.plugin import IPlugin
from twisted.python import log
from twisted.internet import threads, defer

from sdfs.interfaces import IURIHandler, ICrontab
from sdfs.exceptions import FileOrFolderNotFoundException
from sdfs.enum import DatabaseType
from sdfs.resource import PluginResource


def touch(fname, times=None):
    with file(fname, 'a'):
        os.utime(fname, times)

def unpack_folder(path, result_path, unpack_all=False):
    already_unpacked = False
    for item in os.listdir(path):
        item_path = os.path.join(path, item)

        if os.path.isfile(item_path):
            if os.path.splitext(item)[1].lower() == '.rar' and (not already_unpacked or unpack_all):
                process = subprocess.Popen(['unrar', 'e', '-o-', '-y', item_path, result_path], stdout=subprocess.PIPE)
                os.waitpid(process.pid, 0)
                already_unpacked = True

        elif os.path.isdir(item_path):
            if re.match('(?i)^cd[1-3]$', item) or re.match('(?i)^sub(s|(titles?))?$', item):
                try:
                    unpack_folder(item_path, result_path, bool(re.match('(?i)^sub(s|(titles?))?$', item)))
                except:
                    log.msg('Unable to unpack subs, skipping')


def merge_subtitles(path, subfiles):
    add_delay = timedelta()
    file_extra = 0
    subfile = {}
    metadata = []
    first_iteration = True
    result_sub = open(os.path.join(path, '__tmp.sub'), 'wb')

    def time_text_to_num(stamp):
        stamp = map(int, stamp.split(':')[:4])
        return timedelta(hours=stamp[0], minutes=stamp[1], seconds=stamp[2], milliseconds=stamp[3])

    for incsubfile in subfiles:
        subid = None

        for line in open(os.path.join(path, incsubfile)):
            line = line.strip()
            if not len(line) or line[0] == '#':
                continue

            if line.split(' ')[0] == 'id:':
                subid = line.split(':')[2].strip()
                if subid not in subfile:
                    subfile[subid] = {'lines': [], 'lang': line.split(':')[1].strip().strip(',')}

            elif not subid:
                if first_iteration:
                    metadata.append(line)

            elif line.split(' ')[0] == 'timestamp:':
                t = line.replace(',', '').split(' ')
                subfile[subid]['lines'].append((time_text_to_num(t[1])+add_delay, struct.unpack('>I', t[3][-8:].decode('hex'))[0] + file_extra))

        add_delay += extractMetadata(createParser(unicode(os.path.join(path, '%s.avi' % os.path.splitext(incsubfile)[0])))).get('duration')
        file_extra += os.path.getsize(os.path.join(path, os.path.splitext(incsubfile)[0]) + '.sub')
        result_sub.write(open(os.path.join(path, os.path.splitext(incsubfile)[0] + '.sub'), 'rb').read())
        first_iteration = False

    result_idx = open(os.path.join(path, '__tmp.idx'), 'w')
    result_idx.write('# VobSub index file, v7 (do not modify this line!)\n')

    for line in metadata:
        result_idx.write(line + '\n')

    for subindex, subdata in sorted(subfile.items()):
        result_idx.write('id: %s, index: %s\n' % (subdata['lang'], subindex))
        for td, fl in subdata['lines']:
            result_idx.write('timestamp: %s, filepos: %s\n' % (
                '%.2i:%.2i:%.2i:%.3i' % (td.seconds / 3600, (td.seconds / 60) % 60, td.seconds % 60, td.microseconds / 1000),
                ('%9s' % hex(fl)[2:]).replace(' ', '0')
            ))

    result_idx.close()
    result_sub.close()

STREAMABLE_ENDINGS = ['mkv', 'mp4', 'iso']

def create_stream(source_folder, target_folder):
    folder_listing = os.listdir(source_folder)
    
    filetypes = set(map(lambda x:x.split('.')[-1].lower(), filter(lambda x:'.' in x, folder_listing)))
    
    if filter(lambda x:x in STREAMABLE_ENDINGS, filetypes):
        return sorted(map(lambda x:os.path.join(source_folder, x),filter(lambda x:x.split('.')[-1].lower() in STREAMABLE_ENDINGS, folder_listing)), key=lambda x:os.path.getsize(x), reverse=True)[0]
    
    else:
        result_file = os.path.join(target_folder, '%s.mkv' % os.path.split(source_folder.rstrip('/'))[1])
        
        if os.path.isfile(result_file):
            touch(result_file)
            return result_file

        if 'avi' in filetypes:
            do_subs_and_cleanup = False
            avi_merge = map(lambda x:os.path.join(source_folder, x), sorted(filter(lambda x:x.lower().endswith('.avi'), folder_listing), key=lambda x:x.lower()))
   
        else:
            rarfiles = sorted(filter(lambda x:x.lower().endswith('rar'), folder_listing), key=lambda x:x.lower())
            if rarfiles:
                useful_rar_file = os.path.join(source_folder, rarfiles[0])
                rf = rarfile.RarFile(useful_rar_file)
                if rf.infolist()[0].filename.split('.')[-1].lower() in STREAMABLE_ENDINGS:
                    return RarObject(useful_rar_file, rf.infolist()[0].filename)
            
            temp_folder = os.path.join(target_folder, str(uuid.uuid4()))
            
            os.mkdir(temp_folder)
            unpack_folder(source_folder, temp_folder)
            unpack_folder(temp_folder, temp_folder, True)
        
            items = os.listdir(temp_folder)
            if filter(lambda x:os.path.splitext(x)[1].lower().lstrip('.') in STREAMABLE_ENDINGS, items):
                # we could check if there are subtitles to merge into it here
                os.rename(os.path.join(temp_folder, filter(lambda x:os.path.splitext(x)[1].lower().lstrip('.') in STREAMABLE_ENDINGS, items)[0]), result_file)
                shutil.rmtree(temp_folder)
                return result_file
        
            avis = sorted(filter(lambda x:os.path.splitext(x)[1].lower() == '.avi', items), key=lambda x:x.lower())
            if not avis:
                # no avis found, better bail
                return None
            
            avi_merge = map(lambda x:os.path.join(temp_folder, x), avis)
            do_subs_and_cleanup = True
    
        exe_cmd = ['mkvmerge', '-o', result_file]
        first_avi = True
        for avi in avi_merge:
            if first_avi:
                exe_cmd.append(avi)
                first_avi = False
            else:
                exe_cmd.append('+%s' % avi)
    
        if do_subs_and_cleanup:
            subs = sorted(filter(lambda x:os.path.splitext(x)[1].lower() == '.idx', items), key=lambda x:x.lower())
        
            if len(avis) == len(subs) and not set(map(lambda x:os.path.splitext(x)[0].lower(), avis)) - set(map(lambda x:os.path.splitext(x)[0].lower(), subs)):
                if len(avis) > 1:
                    merge_subtitles(temp_folder, subs)
                    exe_cmd.append(os.path.join(temp_folder, '__tmp.idx'))
                else:
                    exe_cmd.append(os.path.join(temp_folder, subs[0]))
    
        process = subprocess.Popen(exe_cmd, stdout=subprocess.PIPE)
        os.waitpid(process.pid, 0)
        
        if do_subs_and_cleanup:
            shutil.rmtree(temp_folder)
        
        return result_file


class RarObject(object):
    def __init__(self, rarpath, filename):
        self.rarpath = rarpath
        self.filename = filename
    
    def get_object(self):
        rf = rarfile.RarFile(self.rarpath)
        ri = rf.infolist()[0]
        return rf.open(ri), ri.file_size


class StreamURIHandler(PluginResource):
    implements(IPlugin, IURIHandler, ICrontab)

    name = 'stream'
    scheme = 'stream'

    @defer.inlineCallbacks
    def render_POST(self, request):
        path = self.get_path(request)
        f = self.filesystem.list_dir(path).values()[0]
        
        if f:
            real_path = os.path.split(self.filesystem.get_file(path.rstrip('/') + '/' + filter(lambda x:x['type'] == DatabaseType.FILE, f)[0]['name']))[0]
            f = yield threads.deferToThread(create_stream, real_path, self.configfile.get(self.config_section, 'resultpath'))
            
            if not f:
                raise FileOrFolderNotFoundException("'%s' not found" % path)
        
        else:
            f = self.filesystem.get_file(path)
            
            if f is None:
                raise FileOrFolderNotFoundException("'%s' not found" % path)
        
        if 'ip' in request.args:
            ip = request.args['ip'][0]
        else:
            ip = None

        defer.returnValue({'url': httpurihandler.prepare_serve(f, ip)})

    def initialize(self):
        pass
    
    def cleanup_unpack_folder(self):
        threads.deferToThread(self._cleanup_unpack_folder)
    
    def _cleanup_unpack_folder(self):
        resultpath = self.configfile.get(self.config_section, 'resultpath')
        log.msg('Cleaning up unpack folder: %s' % resultpath)
        for item in os.listdir(resultpath):
            item = os.path.join(resultpath, item)
            
            if datetime.fromtimestamp(os.path.getmtime(item)) > datetime.now() - timedelta(days=14):
                continue
            
            if os.path.isdir(item) and re.match(r'.+/[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$', item):
                log.msg('Removing stalled unpack folder: %s' % item)
                shutil.rmtree(item)
            
            if os.path.isfile(item) and os.path.splitext(item)[1] == '.mkv':
                log.msg('Removing old streamable: %s' % item)
                os.remove(item)
    
    def schedule(self):
        if self.configfile.has_option(self.config_section, 'cleanup') and self.configfile.get_bool(self.config_section, 'cleanup'):
            cleanup_timestamp = datetime.now() + timedelta(hours=1)
            return '* * * %s %s' % (cleanup_timestamp.hour, cleanup_timestamp.minute), self.cleanup_unpack_folder

streamurihandler = StreamURIHandler()

from sdfs.plugins.http import httpurihandler