'''
Python module to read, play, and write sound data.
For flexibility, FFMPEG is used for non-WAV files.

You can obtain it for free from
    http://ffmpeg.org
    
Mac users should best get binaries from
    http://www.evermeet.cx/ffmpeg/ffmpeg-2.1.4.7z
    http://www.evermeet.cx/ffplay/ffplay-2.1.4.7z

Note that FFMPEG must be installed externally!
Please install ffmpeg/ffplay in the following directory:

Windows: C:\Program Files\ffmpeg\bin\
Mac:	/usr/local/bin/ (is already included in the default paths of the Mac terminal.)
Linux:	/usr/bin/

Compatible with Python 2.x and 3.x
'''

'''
Date:   March 2014
Ver:    1.7
Author: thomas haslwanter

Changes: 1.2 replaced Qt with wxpython, because of timer problems
Changes: 1.3 put ui into thLib; allow "cancel" for "writeWav"
Changes: 1.4 Use FFMPEG for conversion from miscellaneous inputs to ".wav", and FFPLAY to play files non-Windows platforms
Changes: 1.5 replace ui with a version that only uses Tkinter, make it compatible with Python 3.x
Changes: 1.6 if FFMPEG is not found in the default location, the user can locate it interactively.
Changes: 1.7 should make it compatible with Linux and Mac.
'''

import os
import sys
from scipy.io.wavfile import read, write
import tempfile
from subprocess import call
from thLib import ui

# "ffmpeg" has to be installed externally, ideally into the location listed below
if sys.platform == 'win32':
    ffmpeg = r'C:\Progra~2\ffmpeg\bin\ffmpeg.exe'
    ffplay = r'C:\Progra~2\ffmpeg\bin\ffplay.exe'
    
elif sys.platform == 'darwin':
    ffmpeg = r'/usr/local/bin/ffmpeg'
    ffplay = r'/usr/local/bin/ffplay'
    
elif sys.platform == 'linux2':
    ffmpeg = r'/usr/bin/ffmpeg'
    ffplay = r'/usr/bin/ffplay'
    
else:
    print('Sorry, only Windows, Mac, and Linux are supported.')
    
    
"""
def module_exists(module_name):
    '''Check if a module exists'''
    if globals().get(module_name, False):
        return True
    return False
"""
class Sound:
    '''
    Class for working with sound in Python.
    '''
    
    def __init__(self, inFile = None, inData = None, inRate = None):
        # Write the input data to an object, or select and get data from a file.
        '''
        Initialize a sound object

        Parameters
        ----------
        inFile : string
            path- and file-name of infile, if you get the sound from a file.
        inData: array
            manually generated sound data; requires "inRate" to be set, too.
        inRate: integer
            sample rate; required if "inData" are entered.

        Returns
        -------
        None :
            No return value. Initializes the Sound-properties.

        Notes
        -----
        For non WAV-files, the file is first converted to WAV using
        FFMPEG, and then read in. A warning is generated, to avoid
        unintentional deletion of existing WAV-files.

        Examples
        --------
        >>> mySound = Sound()

        '''

        if not inData == None:
            self.setData(inData, inRate)
        else: 
            if inFile == None:
                inFile = self._selectInput()
            if os.path.exists(inFile):    
                self.source = inFile
                self.readSound(self.source)
            else:
                print(inFile + ' does NOT exist!')
        
    def readSound(self, inFile):
        '''
        Read data from a sound-file.

        Parameters
        ----------
        inFile : string
            path- and file-name of infile

        Returns
        -------
        None :
            No return value. Sets the property "data" of the object.

        Notes
        -----
        For non WAV-files, the file is first converted to WAV using
        FFMPEG, and then read in.

        Examples
        --------
        >>> mySound = Sound()
        >>> mySound.readSound('test.wav')

        '''
        
        # Python can natively only read "wav" files. To be flexible, use "ffmpeg" for conversion for other formats
        
        global ffmpeg   # Since "ffmpeg" is modified, it has to be declared "global"
        
        if not os.path.exists(ffmpeg):
            (ffmpeg_file, ffmpeg_path) = ui.getfile(DialogTitle='Please locate the "ffmpeg" executable: ' )
            ffmpeg = os.path.join(ffmpeg_path, ffmpeg_file)
            
        (root, ext) = os.path.splitext(inFile)
        if ext[1:].lower() != 'wav':
            outFile = root + '.wav'
            cmd = [ffmpeg, '-i', inFile, outFile]
            call(cmd, shell=True)
            print('Infile converted from ' + ext + ' to ".wav"')
            
            inFile = outFile
            self.source = outFile

        try:
            self.rate, self.data = read(inFile)
            if len(self.data.shape)==2:     # for stereo, take the first channel
                self.data = self.data[:,0]
            print('data read in!')
        except IOError:
            print('Could not read ' + inFile)
    
    def play(self):
        '''
        Play a sound-file.

        Parameters
        ----------
        None : 

        Returns
        -------
        None :
            

        Notes
        -----
        On "Windows" the module "winsound" is used; on other
        platforms, the sound is played using "ffplay" from FFMPEG.

        Examples
        --------
        >>> mySound = Sound()
        >>> mySound.readSound('test.wav')
        >>> mySound.play()

        '''

        global ffplay
        
        if sys.platform=='win32':
            import winsound
            
        try:
            if self.source == None:
                # If no corresponding sound-file exists, create a temporary file that you can play with an external player
                tmpFile = tempfile.TemporaryFile(suffix='.wav')
                self.writeWav(tmpFile)
                playFile = tmpFile
                
            elif os.path.exists(self.source):
                playFile = self.source
                
            else:
                print('{0} does not exist'.format(self.source))
                return
                
            if sys.platform=='win32':
                winsound.PlaySound(playFile, winsound.SND_FILENAME)
            else:
                if not os.path.exists(ffplay):
                    (ffplay_file, ffplay_path) = ui.getfile(DialogTitle='Please locate "ffplay" executable: ' )
                    ffplay = os.path.join(ffplay_path, ffplay_file)
                
                cmd = [ffplay, '-autoexit', '-nodisp', '-i', playFile]
                call(cmd)
                print('Playing ' + playFile)
            
        except SystemError:
            print('If you don''t have FFMPEG available, you can e.g. use installed audio-files. E.g.:')
            print('import subprocess')
            print('subprocess.call([r"C:\Program Files (x86)\VideoLAN\VLC\vlc.exe", r"C:\Music\14_Streets_of_Philadelphia.mp3"])')
            
    def setData(self, data, rate):
        ''' Set the properties of a Sound-object. '''

        self.data = data
        self.rate = rate
        self.source = None

    def writeWav(self, fullOutFile = None):            
        '''
        Write sound data to a WAV-file.

        Parameters
        ----------
        fullOutFile : string
            Path- and file-name of the outfile. If none is given,
            the user is asked interactively to choose a folder/name
            for the outfile.

        Returns
        -------
        None :
            

        Examples
        --------
        >>> mySound = Sound()
        >>> mySound.readSound('test.wav')
        >>> mySound.writeWav()

        '''

        if fullOutFile == None:
            (outFile , outDir) = ui.savefile('Write sound to ...', '*.wav')            
            if outFile == 0:
                print('Output discarded.')
                return 0
            else:
                fullOutFile = os.path.join(outDir, outFile)
        else:
            outDir = os.path.abspath(os.path.dirname(fullOutFile))
            outFile = os.path.basename(fullOutFile)
        #        fullOutFile = tkFileDialog.asksaveasfilename()

        write(str(fullOutFile), int(self.rate), self.data)
        print('Sounddata written to ' + outFile + ', with a sample rate of ' + str(self.rate))
        print('OutDir: ' + outDir)

        return fullOutFile

    def _selectInput(self):
        '''GUI for the selection of an in-file. '''

        (inFile, inPath) = ui.getfile('*.wav;*.mp3', 'Select sound-input: ')
        fullInFile = os.path.join(inPath, inFile)
        print('Selection: ' + fullInFile)
        return fullInFile


def main():
    # Main function, to test the module
    inSound = Sound()
    inSound.play()
    inSound.writeWav()
    
if __name__ == '__main__':
    main()
    
