'''
Miscellaneous user interface utilities for selecting files or directories.
If nothing or a non-existing file/direcoty is selected, the return is "0". 
Otherwise the file/directory is returned.
'''

'''
ThH, Jan 2014
Ver 3.0
'''


import sys
try:
    # Python 2.x
    import Tkinter
    import tkFileDialog as tkf
except ImportError:
    # Python 3.x
    import tkinter as Tkinter
    import tkinter.filedialog as tkf
    
import os
import matplotlib.pyplot as plt
from numpy import sum, diff, r_, sqrt

def getfile(FilterSpec='*', DialogTitle='Select File: ', DefaultName=''):
    '''
    Selecting an existing file.
    
    Parameters
    ----------
    FilterSpec : query-string
        File filters
    DialogTitle : string
        Window title
    DefaultName : string
        Can be a directory AND filename
    
    Returns
    -------
    filename :  string
        selected existing file
    pathname:   string
        selected path
    
    Examples
    --------
    >>> (myFile, myPath) = thLib.ui.getfile('*.py', 'Testing file-selection', 'c:\\temp\\test.py')
    
    '''
    
    root = Tkinter.Tk()
    root.withdraw()
    fullInFile = tkf.askopenfilename(initialfile=DefaultName, title=DialogTitle, filetypes=[('all files','*'), ('Select', FilterSpec
                                                                                                                          )])
    if not os.path.exists(fullInFile):
        return (0, 0)
    else:
        print('Selection: ' + fullInFile)
        dirName = os.path.dirname(fullInFile)
        fileName = os.path.basename(fullInFile)
        return (fileName, dirName)
        
def savefile(FilterSpec='*',DialogTitle='Save File: ', DefaultName=''):
    '''
    Selecting an existing or new file:
    
    Parameters
    ----------
    FilterSpec : string
        File filters.
    DialogTitle : string
        Window title.
    DefaultName : string
        Can be a directory AND filename.
    

    Returns
    -------
    filename : string
        Selected file.
    pathname : string
        Selecte path.
    

    Examples
    --------
    >>> (myFile, myPath) = thLib.ui.savefile('*.py', 'Testing file-selection', 'c:\\temp\\test.py')

    '''
    
    root = Tkinter.Tk()
    root.withdraw()
    fullOutFile = tkf.asksaveasfile(mode='w', title=DialogTitle, initialfile=DefaultName, filetypes=[('Save as', FilterSpec)]).name
    print('Selection: ' + fullOutFile)
    dirName = os.path.dirname(fullOutFile)
    fileName = os.path.basename(fullOutFile)
    return (fileName, dirName)

def getdir(DialogTitle='Select Directory', DefaultName='.'):
    ''' Select a directory
    
    Parameters
    ----------
    DialogTitle : string
        Window title
    DefaultName : string
        Can be a directory AND filename

    
    Returns
    -------
    directory : string
        Selected directory.

    
    Examples
    --------
    >>> myDir = thLib.ui.getdir('c:\\temp', 'Pick your directory')
    
    '''
    
    root = Tkinter.Tk()
    root.withdraw()
    fullDir = tkf.askdirectory(initialdir=DefaultName, title=DialogTitle)
    if not os.path.exists(fullDir):
        return 0
    else:
        print('Selection: ' + fullDir)
        return fullDir
        
class Viewer:
    '''Interactive navigation through a plot, given an axis handle.

    ::

        X-AXIS
        "n"ext ... full frame forward
        "f"orward ... half frame forward
        "b"ackward ... half frame backward
        "p"revious ... one frame backward
        "z"oom ... zoom in to 10% of the maximum x-range
        
        Y-AXIS
        "u"p ... half frame up
        "d"own ... half frame down
        "c"enter ... center data with the current y-range about y=0
        
        BOTH AXES
        "i"n ... zoom in
        "o"ut ... zoom out
        "a"ll ... show all data
        e"x"it ... close the current axis 
    
    Parameters
    ----------
    ax : axis handle

    Examples
    --------
    >>> plot(np.arange(10))
    >>> thLib.ui.Viewer(gca())
    
    '''
    
    def __init__(self, ax):
        '''Get the required parameters, and display help'''
        
        info_txt = '''This utility lets you navigate with keyboard through data in the "current axis":
        
        X-AXIS
        "n"ext ... full frame forward
        "f"orward ... half frame forward
        "b"ackward ... half frame backward
        "p"revious ... one frame backward
        "z"oom ... zoom in to 10% of the maximum x-range
        
        Y-AXIS
        "u"p ... half frame up
        "d"own ... half frame down
        "c"enter ... center data with the current y-range about y=0
        
        BOTH AXES
        "i"n ... zoom in
        "o"ut ... zoom out
        "a"ll ... show all data
        e"x"it ... close the current axis '''
        print(info_txt)
        
        self.ax = ax
        self.fig = ax.get_figure()
        
        # Get the total data range
        self.xrange_total = ax.xaxis.get_data_interval()
        self.yrange_total = ax.yaxis.get_data_interval()
        
        # Get the current axis limits
        self.xlim = ax.get_xlim()
        self.ylim = ax.get_ylim()
        
        # Get the current axis range
        self._xrange = max(self.xlim) - min(self.xlim)
        self._yrange = max(self.ylim) - min(self.ylim)
        
        # key -> value mapping, for keyboard navigation
        self._hor = dict(p=-1., b=-0.5, z=0, f=0.5, n=1.)
        self._ver = dict(u=0.5, c=0, d=-0.5)
        self._combined = dict(i=1./sqrt(2), a=0, o=sqrt(2))
        self._terminate = 'x'
        
        # Disable the default "f"-mapping
        plt.rcParams['keymap.fullscreen']='ctrl+f'
        plt.rcParams['keymap.all_axes']=''
        plt.rcParams['keymap.zoom']=''
        
        plt.connect('key_press_event', self.OnKeyPress)
        self.fig.canvas.draw()
        plt.pause(0.1)
        
    def set_xlim(self, factor):
        '''Change plot limits of x-axis. The view is limited by the data range.
        
        Parameters
        ----------
        factor : float
            Fraction of current range, for moving the plot forward/backward.
            
        '''
        
        # Limit the views to the lower/upper limit of the x-axis.
        
        if factor>0:
            x_upper = min( (max(self.xrange_total), self.xlim[1]+factor*self._xrange) )
            x_lower = x_upper - self._xrange
            
        elif factor<0:
            x_lower = max( (min(self.xrange_total), self.xlim[0]+factor*self._xrange) )
            x_upper = x_lower + self._xrange
            
        else:   # zoom in, to 10% of the maximum range
            x_lower = min(self.xrange_total)
            self._xrange = 0.1*(self.xrange_total[1]-self.xrange_total[0])
            x_upper = x_lower + self._xrange
           
        # Set the limit
        self.ax.set_xlim([x_lower, x_upper])
        self.xlim = self.ax.get_xlim()
        
        # This one is important! "show" and "draw" don't do the trick here.
        plt.pause(0.1)            
        
    def set_ylim(self, factor):
        '''Change plot limits of y-axis. The view is limited by the data range.
        
        Parameters
        ----------
        factor : float
            Fraction of current range, for moving the plot up/down.
            "0" centers the plot around y=0.
            
        '''
        
        # Limit the views to the lower/upper limit of the y-axis.
        if factor>0:
            y_upper = min( (max(self.yrange_total), self.ylim[1]+factor*self._yrange) )
            y_lower = y_upper - self._yrange
            
        elif factor<0:
            y_lower = max( (min(self.yrange_total), self.ylim[0]+factor*self._yrange) )
            y_upper = y_lower + self._yrange
        
        else:   # center view around y=0
            y_lower = -self._yrange/2.
            y_upper =  self._yrange/2.
            
        # Set the limit
        self.ax.set_ylim([y_lower, y_upper])
        self.ylim = self.ax.get_ylim()
        
        # This one is important! "show" and "draw" don't do the trick here.
        plt.pause(0.1)            
        
    def set_bothlim(self, factor):
        '''Change plot limits of x- and y-axis. The view is limited by the data range.
        
        Parameters
        ----------
        factor : float
            Fraction of current range, for zooming in/out
            "0" shows all data.
            
        '''
        
        x_center = sum(self.xlim)/2.
        y_center = sum(self.ylim)/2.
        
        self._xrange *= factor
        self._yrange *= factor
        
        self.xlim = x_center + self._xrange*r_[-0.5,0.5]
        self.ylim = y_center + self._yrange*r_[-0.5,0.5]
        
        if factor>0:
            x_lower = max(self.xrange_total[0], self.xlim[0])
            x_upper = min(self.xrange_total[1], self.xlim[1])
            y_lower = max(self.yrange_total[0], self.ylim[0])
            y_upper = min(self.yrange_total[1], self.ylim[1])
            
        elif factor==0:
            x_lower, x_upper = self.xrange_total
            y_lower, y_upper = self.yrange_total
            
            self._xrange = float(diff(self.xrange_total))
            self._yrange = float(diff(self.yrange_total))
            
        # Set the limits
        self.ax.set_xlim([x_lower, x_upper])
        self.ax.set_ylim([y_lower, y_upper])
        self.xlim = self.ax.get_xlim()
        self.ylim = self.ax.get_ylim()
        
        # This one is important! "show" and "draw" don't do the trick here.
        plt.pause(0.1)            
        
    def OnKeyPress(self, event):
        '''Define the keyboard action'''

        #print('press', event.key)
        
        hor = self._hor
        ver = self._ver
        both = self._combined
        
        # If you do anything but a horizontal/vertical navigation or zooming,exit the plot
        selection = event.key
        if selection not in (hor.keys() + ver.keys() + both.keys()):
            selection = 'x'
        
        # Horizontal navigation
        if selection in hor.keys():
            self.set_xlim(hor.get(selection))
            
        # Vertical navigation
        elif selection in ver.keys():
            self.set_ylim(ver.get(selection))
        
        # Zoom in/out:
        elif selection in both.keys():
            self.set_bothlim(both.get(selection))
            
        else:
            plt.close(plt.gcf())
            
        # Update plot
        self.fig.canvas.draw()

        
if __name__ == "__main__":   
    # Test functions
    (myFile, myPath) = getfile('*.eps', 'Testing file-selection', r'c:\temp\test.eps')
    if myFile == 0:          
        print(0)
    else:
        print('File: %s, Path: %s' % (myFile, myPath))
    (myFile, myPath) = savefile('*.txt', 'Testing saving-selection', r'c:\temp\test.txt')
        
    myDir = getdir()
    print(myDir)
