'''
Routines for working with vectors
These routines can be used with vectors, as well as with matrices containing a vector in each row.
'''
 
'''
author :  Thomas Haslwanter
date :    Nov-2013
version : 1.0
'''

import numpy as np

def normalize(v):
    ''' Normalization of a given vector 
    
    Parameters
    ----------
    v : array (N,) or (M,N)
        input vector

    Returns
    -------
    v_normalized : array (N,) or (M,N)
        normalized input vector

    Example
    -------
    >>> thLib.vector.normalize([3, 0, 0])
    array([[ 1.,  0.,  0.]])
    
    >>> v = [[pi, 2, 3], [2, 0, 0]]
    >>> thLib.vector.normalize(v)
    array([[ 0.6569322 ,  0.41821602,  0.62732404],
       [ 1.        ,  0.        ,  0.        ]])
    
    '''
    
    v = np.atleast_2d(v)
    length = np.sqrt(np.sum(v**2,axis=1))
    if v.shape[1]==1:
        v = v.flatten()
    return (v.T/length).T

def project(v1,v2):
    '''Project one vector onto another
    
    Parameters
    ----------
    v1 : array (N,) or (M,N)
        projected vector
    v2 : array (N,) or (M,N)
        target vector

    Returns
    -------
    v_projected : array (N,) or (M,N)
        projection of v1 onto v2

    Example
    -------
    >>> v1 = np.array([[1,2,3],
    >>>       [4,5,6]])
    >>> v2 = np.array([[1,0,0],
    >>>       [0,1,0]])
    >>> thLib.vector.project(v1,v2)
    array([[ 1.,  0.,  0.],
       [ 0.,  5.,  0.]])
    
    '''
    
    v1 = np.atleast_2d(v1)
    e2 = normalize(np.atleast_2d(v2))
    return (e2.T * map(np.dot, v1, e2)).T
 
def GramSchmidt(p1,p2,p3):
    '''Gram-Schmidt orthogonalization
    
    Parameters
    ----------
    p1 : array (3,) or (M,3)
        coordinates of Point 1
    p2 : array (3,) or (M,3)
        coordinates of Point 2
    p3 : array (3,) or (M,3)
        coordinates of Point 3

    Returns
    -------
    Rmat : array (9,) or (M,9)
        flattened rotation matrix, corresponding to [e1 e2 e3]

    Example
    -------
    >>> P1 = np.array([[0, 0, 0], [1,2,3]])
    >>> P2 = np.array([[1, 0, 0], [4,1,0]])
    >>> P3 = np.array([[1, 1, 0], [9,-1,1]])
    >>> GramSchmidt(P1,P2,P3)
    array([[ 1.        ,  0.        ,  0.        ,  0.        ,  1.        ,
         0.        ,  0.        ,  0.        ,  1.        ],
       [ 0.6882472 , -0.22941573, -0.6882472 ,  0.62872867, -0.28470732,
         0.72363112, -0.36196138, -0.93075784, -0.05170877]])

    '''
    
    v1 = np.atleast_2d(p2-p1)
    v2 = np.atleast_2d(p3-p1)
        
    e1 = normalize(v1)
    e2 = normalize(v2- project(v2,e1))
    e3 = np.cross(e1,e2)
    
    return np.hstack((e1,e2,e3))