import unittest
import numpy as np
from numpy import sin, cos, array, r_, vstack, abs, tile, pi
from numpy.linalg import norm
import thLib
from time import sleep

class TestSequenceFunctions(unittest.TestCase):

    def test_fit_sin(self):
        np.random.seed(1234)
        t = np.arange(0,10,0.1)
        x = 3 + 4*sin(2*np.pi*1*t + 5*np.pi/180) + np.random.randn(len(t))
        result = array(thLib.fits.fit_sin(t, x, 1))
        correct = array([ 4.70902766,  3.86327971,  3.03511228])
        self.assertAlmostEqual(0, norm(result-correct))
        
    def test_fit_circle(self):
        r = 2
        center = r_[5,5]
        theta = r_[0:2*pi:10j]
        x = r*cos(theta)+center[0]
        y = r*sin(theta)+center[1]
        cFit,rFit = thLib.fits.fit_circle(x,y)
        self.assertAlmostEqual(r, rFit)
        self.assertAlmostEqual(0, norm(center-cFit))
    
    def test_fit_exp(self):
        t = np.arange(10)
        tau = 2.
        amp = 1.
        offset = 2.
        x = offset + amp*np.exp(-t/tau)    
        fitted =  thLib.fits.fit_exp(t,x)
        
        self.assertAlmostEqual(tau, fitted.tau)
        self.assertAlmostEqual(amp, fitted.amplitude)
        self.assertAlmostEqual(offset, fitted.offset)
        
    def test_fit_line(self):
        x = np.array([0, 0, 10, 10])
        y = np.array([1, 3, 11, 13])
        out = thLib.fits.fit_line(x,y)
        
        self.assertAlmostEqual(out.intercept, 2)
        self.assertAlmostEqual(out.slope, 1)
        ciError_intercept = out.CIs[0] - array([-2.30265273,  6.30265273])
        ciError_slope = out.CIs[1] - array([ 0.39151302,  1.60848698])
        self.assertAlmostEqual(0, norm(ciError_intercept))
        self.assertAlmostEqual(0, norm(ciError_slope))
        
    def test_regress(self):
        x = np.array([0, 0, 10, 10])
        y = np.array([1, 3, 11, 13])
        (fit,ci) = thLib.fits.regress(y,x) 
        
        ciError = ci - array([[-2.30265273,  6.30265273],
                 [ 0.39151302,  1.60848698]])
        
        self.assertAlmostEqual(fit[0], 2)
        self.assertAlmostEqual(fit[1], 1)
        self.assertAlmostEqual(norm(ciError), 0)
 
if __name__ == '__main__':
    unittest.main()
    print('Thanks for using programs from Thomas!')
    sleep(2)
