"""
Find the best-fit circle in an image, using the RANSAC algorithm

Author : Thomas Haslwanter
Date : Nov-2012
Version: 1.0
"""

from skimage import filter
import numpy as np
import os
import matplotlib.pyplot as plt

debug_flag = 1

def fitCircle(x,y):
    '''Determine the best-fit circle to given datapoints.'''
    M = np.vstack((2*x,2*y,np.ones(len(x)))).T
    (par,_,_,_) = np.linalg.lstsq(M,x**2+y**2)
    center = par[:2]
    radius = np.sqrt(par[2]+np.sum(center**2))
    return(center, radius)

def drawCircle(center,r):
    '''Draw a circle'''
    nPts = 100
    phi = np.linspace(0,2*np.pi,nPts)
    x = center[0] + r*np.cos(phi)
    y = center[1] + r*np.sin(phi)
    plt.hold(True)
    plt.plot(y,x,'r')

def main():
    '''Main program for the RANSAC fit'''
    # Get the data
    os.chdir(r'C:\Users\p20529\Coding\Matlab\imProc\ransac_fitCircle')
    data = plt.imread('0_5.bmp')
    
    # Eliminate edge artefacts
    rim = 20
    data = data[rim:-rim, rim:-rim]
    imSize = data.shape
    
    # Find edges
    edges = filter.sobel(data)
    edgePnts = edges>0.15
    np.sum(edgePnts)
    [x,y] = np.where(edgePnts==True)
       
    # set RANSAC parameters
    par={'eps': 3, 
         'ransac_threshold': 0.1, 
         'nIter': 500, 
         'lowerRadius': 5,
         'upperRadius': 200   
    }
    
    # Allocate memory, for center(2D), radius, numPoints (structured array)
    fitted = np.zeros(par['nIter'], \
    dtype={'names':['center', 'radius', 'nPts'], 'formats':['2f8', 'f8', 'i4']})
    
    for ii in range(par['nIter']):
        # Takes 3 random points, and find the corresponding circle
        randEdges = np.random.permutation(len(x))[:3]
        (center, radius) = fitCircle(x[randEdges], y[randEdges])
        
        # Eliminate very small and very large circles, and centers outside the image
        if not (par['lowerRadius'] < radius < par['upperRadius'] and \
                0 <= center[0] < imSize[0] and \
                0 <= center[1] < imSize[1]):
            continue

        # Make sure a reasonable number of points lies near that circle
        centerDistance = np.sqrt((x-center[0])**2 + (y-center[1])**2)
        inCircle = np.where(np.abs(centerDistance-radius)<par['eps'])[0]
        inPts = len(inCircle)
        if inPts < par['ransac_threshold'] *4*np.pi*radius*par['eps'] or inPts < 3:
            continue
        
        # Fit a circle to all good points, and save the corresponding parameters
        (center, radius) = fitCircle(x[inCircle], y[inCircle])        
        fitted[ii] = (center, radius, inPts)
    
        # If you want to see these points:
        if debug_flag == 1:
            plt.plot(y,x,'.')
            plt.hold(True)
            plt.plot(y[inCircle], x[inCircle],'r.')
            plt.plot(y[randEdges], x[randEdges], 'g+', ms=15)
            plt.axis('equal')
            plt.show()
        
    # Sort the circles, according to number of points included
    fitted = np.sort(fitted,order='nPts')
        
    # Show the best-fitting circle
    plt.imshow(data, cmap='gray', origin='lower')
    drawCircle(fitted[-1]['center'], fitted[-1]['radius'])
    plt.show()
    
if __name__=='__main__':
    main()
