'''
Author : ThH
Date : Oct 2012
Ver : 1.0
'''

from numpy import pi, sin, cos, arange, array
from numpy.linalg import lstsq, norm
import matplotlib.pyplot as plt
from math import atan2
 
def fitSine(tList,yList,freq):
   ''' Fit a sine wave with a known frequency to a given set of data.
   
   y = amplitude * sin(2*pi*freq * tList + phase*pi/180) + bias
   
   Inputs:
   -------
       yList ... datapoints, as array 
       tList ... time base, in sec
       freq ... in Hz
       
   Returns:
   --------
       phase ... in degrees
       amplitude 
       bias
   '''
   
   # Get input data
   b = yList
   rows = [ [sin(freq*2*pi*t), cos(freq*2*pi*t), 1] for t in tList]
   A = array(rows)
   
   # Make the fit
   (w,residuals,rank,sing_vals) = lstsq(A,b)
   
   # Extract desired parameters ...
   phase = atan2(w[1],w[0])*180/pi
   amplitude = norm([w[0],w[1]],2)
   bias = w[2]
   
   # ... and return them
   return (phase,amplitude,bias)
 
if __name__=='__main__':
   '''Example of the fit-program'''
   import random
 
   # Set the parameters
   tList = arange(0.0,1.0,0.001)
   tSamples = arange(0.0,1.0,0.05)
   random.seed(0.0)
   phase = 65
   amplitude = 3
   bias = -0.3
   frequency = 4
   
   # Generate dummy data
   yList    = amplitude*sin(tList   *frequency*2*pi+phase*pi/180.0)+bias
   ySamples = amplitude*sin(tSamples*frequency*2*pi+phase*pi/180.0)+bias
   yMeasured = [y+random.normalvariate(0,2) for y in ySamples]
   
   # Make the fit
   (phaseEst,amplitudeEst,biasEst) = fitSine(tSamples,yMeasured,frequency)
   
   # Print the results
   print ('Phase estimate = %f, Amplitude estimate = %f, Bias estimate = %f'
       % (phaseEst,amplitudeEst,biasEst))
        
   # Plot the data
   yEst = amplitudeEst*sin(tList*frequency*2*pi+phaseEst*pi/180.0)+biasEst
 
   plt.figure(1)
   plt.plot(tList,yList,'b')
   plt.plot(tSamples,yMeasured,'+r',markersize=12,markeredgewidth=2)
   plt.plot(tList,yEst,'-g')
   plt.xlabel('seconds')
   plt.legend(['True value','Measured values','Estimated value'])
   plt.grid(True)
   plt.show()