''' 
Demo of a non-linear fit, using "optimize" from Scipy 
 
author : Thomas Haslwanter 
date : Sept 2013 
version : 1.1 
''' 
 
import numpy as np 
from scipy import optimize 
import matplotlib.pyplot as plt 
 
def generate_data(tau = 5, offset = 3, amp = 2, noiseAmp = 0.1, rate = 20): 
    ''' Generate the data ''' 
    t = np.r_[0:20:(1./rate)] 
    y = offset + amp*np.exp(-t/tau) + np.random.randn(len(t))*noiseAmp 
     
    start = noiseAmp*np.random.randn(50) + max(y) 
    yLong = np.hstack((start, y)) 
     
    # Plot them 
    plt.plot(np.arange(len(yLong))/float(rate), yLong) 
    plt.show() 
    return (yLong, rate) 
 
def prepare_for_fit(yLong, rate): 
    '''Eliminate a constant offset before the exponential decay.''' 
    threshold = 0.9 
     
    # Find the beginning of the decay 
    tStart = np.min(np.where(yLong < np.max(yLong)*threshold)[0]) 
     
    # Get the data for the fit 
    yFit = yLong[tStart:] 
    tFit = np.arange(len(yFit))/float(rate) 
    return (tFit, yFit) 
 
def makeFit(tFit, yFit): 
    '''Calculate the best-fit parameters for the exponential decay to an offset.''' 
     
    # Define the fit-function and the error-function 
    fitfunc = lambda p, x: p[0] + p[1]*np.exp(-x/p[2]) 
    errfunc  = lambda p,x,y: fitfunc(p,x) - y 
     
    pInit = np.r_[0, 1, 1]  # Initial values 
     
    # Make the fit 
    pFit, success = optimize.leastsq(errfunc, pInit, args=(tFit, yFit)) 
     
    # Plot the data and the fit 
    plt.plot(tFit, yFit, label='rawdata')
    plt.hold(True)
    plt.plot(tFit, fitfunc(pFit,tFit), label='fit') 
    plt.legend()
    plt.show() 

if __name__ == '__main__':
    (yData, rate) = generate_data() 
    (tFit, yFit) = prepare_for_fit(yData, rate) 
    makeFit(tFit, yFit) 
    raw_input('Done')
                                
