#  _________________________________________________________________________
#
#  PyUtilib: A Python utility library.
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  _________________________________________________________________________

__all__ = ['Task', 'EmptyTask', 'reset_task_counter', 'Component']

import connections
from options import OptionParser


def unique_task_id():
    unique_task_id.counter += 1
    return unique_task_id.counter
unique_task_id.counter = 0


def reset_task_counter():
    global unique_task_id
    unique_task_id.counter = 0


class Task(object):

    def __init__(self, id=None, name=None):
        self.id = id if not id is None else unique_task_id()
        if name is None:
            self.name = "Task"+str(self.id)
        else:
            self.name = name
        self.inputs = connections.InputConnections(self)
        self.inputs.set_name(self.name+"-inputs")
        self.outputs = connections.OutputConnections(self)
        self.outputs.set_name(self.name+"-outputs")
        self._resources = []
        self._predecessors = []
        self._create_parser()


    def add_resource(self, res):
        self._resources.append(res)

    def next_tasks(self):
        return set(t for name in self.outputs for t in self.outputs[name].to_task)
            
    def prev_tasks(self):
        return set(self.inputs[name].from_task for name in self.inputs if self.inputs[name].from_task.id != NoTask.id) | set(task for task in self._predecessors)
           
    def next_task_ids(self):
        return set(task.id for task in self.next_tasks())
            
    def prev_task_ids(self):
        return set(task.id for task in self.prev_tasks())
           
    def execute(self):
        raise ValueError, "There is no default execution for an abstract Task object! Task=%s" % str(self) #pragma:nocover

    def ready(self):
        for res in self._resources:
            if not res.available():
                return False
        return True

    def __call__(self, *options, **kwds):
        """ 
        Copy the inputs into this Task's dictionary, 
        then execute the task, then copy 
        the outputs out of the dictionary.
        """
        # Set inputs
        for opt in options:
            self._set_inputs(opt)
        self._set_inputs(kwds)
        #
        for res in self._resources:
            res.lock()
        for i in self.inputs:
            setattr(self, i, self.inputs[i].get_value())
        self.execute()
        for i in self.outputs:
            self.outputs[i].set_value( getattr(self, i) )
        for res in self._resources:
            res.unlock()

    def set_options(self, args):
        [self.options, args] = self.parser.parse_args(args)
        tmp = {}
        for opt in self.parser._long_opt:
            try:
                val = getattr(self.options,opt[2:])
                tmp[opt[2:]] = val
            except:
                pass
        self._set_inputs(tmp)

    def _set_inputs(self, options):
        #print "HERE",options
        for key in options:
            self.inputs[key].set_value(options[key])

    def _create_parser(self):
        """
        Create the OptionParser object and populated it with option groups.
        """
        self.parser = OptionParser()
        self._parser_group = {}
        self._create_parser_groups()
        self.parser.ignore_invalid()
        for key in self._parser_group:
            self.parser.add_option_group(self._parser_group[key])

    def _create_parser_groups(self):
        """
        This method is called by the _create_parser method to setup the
        parser groups that are registered for this task.
        """

    def __repr__(self):
        return str(self)

    def __str__(self):
        return "%s prev: %s next: %s resources: %s" % (str(self.name),str(sorted(list(self.prev_task_ids()))),str(sorted(list(self.next_task_ids()))), str(self._resources))


"""
Alias for the Task class
"""
class Component(Task):

    def __init__(self, *args, **kwds):
        Task.__init__(self, *args, **kwds)


class EmptyTask(Task):

    def __init__(self, id=None, name=None):
        Task.__init__(self, id=None, name=None)

    def __call__(self, *args, **kwds):
        pass

NoTask = Task(id=0)

