'''
Created on 7 jan. 2013

@author: sander
'''
from bitstring import ConstBitStream, BitArray, Bits
from pylisp.packet.lisp.control import type_registry, LISPControlMessage, \
    LISPMapReferralRecord


__all__ = ['LISPMapReferralMessage']


class LISPMapReferralMessage(LISPControlMessage):
    # Class property: which message type do we represent?
    message_type = 6

    def __init__(self, nonce='\x00\x00\x00\x00\x00\x00\x00\x00', records=None):
        '''
        Constructor
        '''
        super(LISPMapReferralMessage, self).__init__()

        # Set defaults
        self.nonce = nonce
        self.records = records or []

    def sanitize(self):
        '''
        Check if the current settings conform to the LISP specifications and
        fix them where possible.
        '''
        super(LISPMapReferralMessage, self).sanitize()

        # WARNING: http://tools.ietf.org/html/draft-ietf-lisp-ddt-00
        # does not define this field so the description is taken from
        # http://tools.ietf.org/html/draft-ietf-lisp-24
        #
        # Nonce: An 8-octet random value created by the sender of the Map-
        # Request.  This nonce will be returned in the Map-Reply.  The
        # security of the LISP mapping protocol depends critically on the
        # strength of the nonce in the Map-Request message.  The nonce
        # SHOULD be generated by a properly seeded pseudo-random (or strong
        # random) source.  See [RFC4086] for advice on generating security-
        # sensitive random data.
        if len(bytes(self.nonce)) != 8:
            raise ValueError('Invalid nonce')

        # Map-Referral Records:  When the M bit is set, this field is the size
        # of a single "Record" in the Map-Reply format.  This Map-Reply record
        # contains the EID-to-RLOC mapping entry associated with the Source
        # EID.  This allows the ETR which will receive this Map-Request to
        # cache the data if it chooses to do so.
        for record in self.records:
            if not isinstance(record, LISPMapReferralRecord):
                raise ValueError('Invalid record')

            record.sanitize()

    @classmethod
    def from_bytes(cls, bitstream):
        '''
        Parse the given packet and update properties accordingly
        '''
        packet = cls()

        # Convert to ConstBitStream (if not already provided)
        if not isinstance(bitstream, ConstBitStream):
            if isinstance(bitstream, Bits):
                bitstream = ConstBitStream(auto=bitstream)
            else:
                bitstream = ConstBitStream(bytes=bitstream)

        # Read the type
        type_nr = bitstream.read('uint:4')
        if type_nr != packet.message_type:
            msg = 'Invalid bitstream for a {0} packet'
            class_name = packet.__class__.__name__
            raise ValueError(msg.format(class_name))

        # Skip reserved bits
        bitstream.read(20)

        # Store the record count
        record_count = bitstream.read('uint:8')

        # Store the nonce
        packet.nonce = bitstream.read('bytes:8')

        # Read the records
        for dummy in range(record_count):
            record = LISPMapReferralRecord.from_bytes(bitstream)
            packet.records.append(record)

        # There should be no remaining bits
        if bitstream.pos != bitstream.len:
            raise ValueError('Bits remaining after processing packet')

        # Verify that the properties make sense
        packet.sanitize()

        return packet

    def to_bytes(self):
        '''
        Create bytes from properties
        '''
        # Verify that properties make sense
        self.sanitize()

        # Start with the type
        bitstream = BitArray('uint:4=%d' % self.message_type)

        # Add padding
        bitstream += BitArray(20)

        # Add the record count
        bitstream += BitArray('uint:8=%d' % len(self.records))

        # Add the nonce
        bitstream += BitArray(bytes=self.nonce)

        # Add the records
        for record in self.records:
            bitstream += record.to_bytes()

        return bitstream.bytes


# Register this class in the registry
type_registry.register_type_class(LISPMapReferralMessage)
