#!/usr/bin/env python
"""
================================================================================
:mod:`main` -- Viewer main class
================================================================================

.. module:: main
   :synopsis: Viewer main class

.. inheritance-diagram:: pyhmsa.gui.viewer.main

"""

# Script information for the file.
__author__ = "Philippe T. Pinard"
__email__ = "philippe.pinard@gmail.com"
__version__ = "0.1"
__copyright__ = "Copyright (c) 2014 Philippe T. Pinard"
__license__ = "MIT"

# Standard library modules.
import os
import re
import sys
import logging
import platform

# Third party modules.
from PySide.QtGui import \
    (QDialog, QHBoxLayout, QScrollArea, QMdiArea, QMainWindow, QDockWidget,
     QAction, QKeySequence, QTreeWidget, QFileDialog, QMessageBox,
     QTreeWidgetItem, QMdiSubWindow, QMenu, QLineEdit, QDialogButtonBox,
     QValidator, QComboBox, QStackedWidget, QFormLayout, QInputDialog,
     QProgressDialog, QApplication, QTabWidget, QVBoxLayout, QWidget,
     QCheckBox)
from PySide.QtCore import Qt

import matplotlib

# Local modules.
from pyhmsa.type.identifier import validate_identifier

from pyhmsa.viewer.controller import Controller

from pyhmsa.gui.util.tango import getIcon
from pyhmsa.gui.spec.header import HeaderWidget
from pyhmsa.gui.util.validation import validate_widget
import pyhmsa.gui.util.messagebox as messagebox
from pyhmsa.gui.util.registry import \
    iter_condition_widgets, iter_datum_widgets, iter_importers, iter_exporters

# Globals and constants variables.
CAMELCASE_TO_WORDS_PATTERN = re.compile('([A-Z][a-z0-9]*)')

matplotlib.use('Qt4Agg')
matplotlib.rcParams['backend.qt4'] = 'PySide'

def _generate_description(identifier, clasz):
    text = identifier + " [" + clasz.TEMPLATE
    if clasz.CLASS is not None:
        text += '/' + clasz.CLASS
    text += ']'
    return text

#--- Validators

class _ConditionValidator(QValidator):

    def __init__(self, controller):
        QValidator.__init__(self)
        self._controller = controller

    def validate(self, text, pos):
        try:
            validate_identifier(text)
        except:
            return QValidator.Intermediate
        if self.controller().hasCondition(text):
            return QValidator.Intermediate
        return QValidator.Acceptable

    def controller(self):
        return self._controller

class _DatumValidator(QValidator):

    def __init__(self, controller):
        QValidator.__init__(self)
        self._controller = controller

    def validate(self, text, pos):
        try:
            validate_identifier(text)
        except:
            return QValidator.Intermediate
        if self.controller().hasDatum(text):
            return QValidator.Intermediate
        return QValidator.Acceptable

    def controller(self):
        return self._controller

#--- Dialogs

class _RenameDialog(QDialog):

    def __init__(self, oldidentifier, controller, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle('Rename')

        # Widgets
        self._txt_identifier = QLineEdit()

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)

        # Layouts
        layout = QHBoxLayout()
        layout.addWidget(self._txt_identifier)
        layout.addWidget(buttons)
        self.setLayout(layout)

        # Signals
        self._txt_identifier.textChanged.connect(self._onTextChanged)
        buttons.accepted.connect(self._onOk)
        buttons.rejected.connect(self._onCancel)

        # Defaults
        self._txt_identifier.setText(oldidentifier)

    def _onTextChanged(self):
        if self._txt_identifier.hasAcceptableInput():
            self._txt_identifier.setStyleSheet("background: none")
        else:
            self._txt_identifier.setStyleSheet("background: pink")

    def _onOk(self):
        if not self._txt_identifier.hasAcceptableInput():
            message = '"%s" is invalid' % self._txt_identifier.text()
            QMessageBox.critical(self, self.windowTitle(), message)
            return
        self.accept()

    def _onCancel(self):
        self.reject()

    def setValidator(self, validator):
        self._txt_identifier.setValidator(validator)

    def validator(self):
        return self._txt_identifier.validator()

    def newIdentifier(self):
        return self._txt_identifier.text()

class _ConditionNewDialog(QDialog):

    def __init__(self, controller, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle('New condition')

        # Widgets
        self._txt_identifier = QLineEdit()
        self._txt_identifier.setValidator(_ConditionValidator(controller))

        self._combobox = QComboBox()
        self._stack = QStackedWidget()

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)

        # Layouts
        layout = QFormLayout()
        layout.addRow('Identifier', self._txt_identifier)
        layout.addRow(self._combobox)
        layout.addRow(self._stack)
        layout.addRow(buttons)
        self.setLayout(layout)

        # Signals
        self._txt_identifier.textChanged.connect(self._onTextChanged)
        self._txt_identifier.textChanged.emit('')
        self._combobox.currentIndexChanged.connect(self._onComboBox)
        buttons.accepted.connect(self._onOk)
        buttons.rejected.connect(self._onCancel)

        # Defaults
        for _, widget in iter_condition_widgets():
            self._combobox.addItem(widget.accessibleName())

            scrollarea = QScrollArea()
            scrollarea.setWidgetResizable(True)
            scrollarea.setWidget(widget)
            self._stack.addWidget(scrollarea)

    def _onTextChanged(self):
        if self._txt_identifier.hasAcceptableInput():
            self._txt_identifier.setStyleSheet("background: none")
        else:
            self._txt_identifier.setStyleSheet("background: pink")

    def _onComboBox(self):
        current_index = self._combobox.currentIndex()
        self._stack.setCurrentIndex(current_index)

    def _onOk(self):
        if not self._txt_identifier.hasAcceptableInput():
            message = 'A condition named "%s" is invalid' % self._txt_identifier.text()
            QMessageBox.critical(self, self.windowTitle(), message)
            return
        if not validate_widget(self._stack.currentWidget().widget()):
            return
        self.accept()

    def _onCancel(self):
        self.reject()

    def identifier(self):
        return self._txt_identifier.text()

    def condition(self):
        return self._stack.currentWidget().widget().parameter()

class _PreferencesTabWidget(QWidget):

    def __init__(self, parent=None):
        QWidget.__init__(self, parent)

    def load(self, settings):
        pass

    def save(self, settings):
        pass

class _PreferencesImportTabWidget(_PreferencesTabWidget):

    def __init__(self, parent=None):
        QWidget.__init__(self, parent)

        # Widgets
        self._chk_search_extra = QCheckBox("Search for extra data file after import")

        # Layouts
        layout = QVBoxLayout()
        layout.addWidget(self._chk_search_extra)
        self.setLayout(layout)

    def load(self, settings):
        _PreferencesTabWidget.load(self, settings)

        value = settings.valueBool("search_extra", True)
        self._chk_search_extra.setChecked(value)

    def save(self, settings):
        _PreferencesTabWidget.save(self, settings)
        settings.setValue("search_extra", self._chk_search_extra.isChecked())

class _PreferencesDialog(QDialog):

    def __init__(self, controller, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle("Settings")

        # Variables
        self._controller = controller

        # Widgets
        self._wdg_tab = QTabWidget()
        self._wdg_tab.setMovable(False)
        self._wdg_tab.setTabsClosable(False)

        self._wdg_tab.addTab(_PreferencesImportTabWidget(), "Import")

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)

        # Layouts
        layout = QVBoxLayout()
        layout.addWidget(self._wdg_tab)
        layout.addWidget(buttons)
        self.setLayout(layout)

        # Signals
        buttons.accepted.connect(self._onOk)
        buttons.rejected.connect(self._onCancel)

        # Defaults
        settings = self.controller().settings()
        for widget in map(self._wdg_tab.widget, range(self._wdg_tab.count())):
            widget.load(settings)

    def _onOk(self):
        settings = self.controller().settings()
        for widget in map(self._wdg_tab.widget, range(self._wdg_tab.count())):
            widget.save(settings)
        self.accept()

    def _onCancel(self):
        self.reject()

    def controller(self):
        return self._controller

#--- Tree items

class _ActionTreeItem(QTreeWidgetItem):

    def __init__(self, controller, parent):
        QTreeWidgetItem.__init__(self, parent)

        # Variables
        self._controller = controller
        self._menu = QMenu()

    def controller(self):
        return self._controller

    def setActions(self, actions):
        self._menu.clear()
        self._menu.addActions(actions)

    def setDefaultAction(self, action):
        self._menu.setDefaultAction(action)

    def popupMenu(self):
        return self._menu

class _HeaderTreeItem(_ActionTreeItem):

    def __init__(self, controller, parent):
        _ActionTreeItem.__init__(self, controller, parent)
        self.setText(0, 'Header')

        # Actions
        action_open = QAction('Open', self.treeWidget())
        self.setActions([action_open])
        self.setDefaultAction(action_open)

        # Signals
        action_open.triggered.connect(self._onOpen)

    def _onOpen(self):
        self.controller().headerOpen.emit()

class _ConditionsTreeItem(_ActionTreeItem):

    def __init__(self, controller, parent):
        _ActionTreeItem.__init__(self, controller, parent)
        self.setText(0, 'Conditions')

        # Actions
        action_add = QAction('Add condition', self.treeWidget())
        self.setActions([action_add])

        # Signals
        action_add.triggered.connect(self._onAdd)

    def _onAdd(self):
        dialog = _ConditionNewDialog(self.controller())
        if dialog.exec_():
            identifier = dialog.identifier()
            condition = dialog.condition()
            self.controller().conditionAdd.emit(identifier, condition)

class _ConditionTreeItem(_ActionTreeItem):

    def __init__(self, identifier, controller, parent):
        _ActionTreeItem.__init__(self, controller, parent)

        # Actions
        action_open = QAction('Open', self.treeWidget())
        sep = QAction(self.treeWidget())
        sep.setSeparator(True)
        action_delete = QAction(getIcon('edit-delete'), 'Delete', self.treeWidget())
        action_rename = QAction('Rename', self.treeWidget())
        self.setActions([action_open, sep, action_delete, action_rename])
        self.setDefaultAction(action_open)

        # Signals
        action_open.triggered.connect(self._onOpen)
        action_delete.triggered.connect(self._onDelete)
        action_rename.triggered.connect(self._onRename)

        # Defaults
        self.setIdentifier(identifier)

    def _onOpen(self):
        self.controller().conditionOpen.emit(self.identifier())

    def _onDelete(self):
        self.controller().conditionDelete.emit(self.identifier())

    def _onRename(self):
        controller = self.controller()
        oldidentifier = self.identifier()

        dialog = _RenameDialog(oldidentifier, controller)
        dialog.setValidator(_ConditionValidator(controller))
        if dialog.exec_():
            newidentifier = dialog.newIdentifier()
            controller.conditionRename.emit(oldidentifier, newidentifier)

    def setIdentifier(self, identifier):
        self._identifier = identifier

        condition_class = type(self.controller().condition(identifier))
        text = _generate_description(identifier, condition_class)
        self.setText(0, text)

    def identifier(self):
        return self._identifier

class _DatumTreeItem(_ActionTreeItem):

    def __init__(self, identifier, controller, parent):
        _ActionTreeItem.__init__(self, controller, parent)

        # Actions and signals
        datum = controller.datum(identifier)
        datum_class_name = type(datum).__name__

        actions_open = {}

        for name, _ in iter_datum_widgets():
            class_name, modifier = name.split('.', 2)
            if class_name != datum_class_name:
                continue

            modifier_name = ' '.join(w.lower() for w in CAMELCASE_TO_WORDS_PATTERN.split(modifier) if w)

            action = QAction('Open %s' % modifier_name, self.treeWidget())
            f = lambda s = self, m = modifier: s.controller().datumOpen.emit(s.identifier(), m)
            action.triggered.connect(f)

            actions_open[modifier_name] = action

        # Actions
        actions = []

        keys = sorted(actions_open.keys())
        for key in keys:
            actions.append(actions_open[key])

        sep = QAction(self.treeWidget())
        sep.setSeparator(True)
        actions.append(sep)

        action_delete = QAction(getIcon('edit-delete'), 'Delete', self.treeWidget())
        action_delete.triggered.connect(self._onDelete)
        actions.append(action_delete)

        action_rename = QAction('Rename', self.treeWidget())
        action_rename.triggered.connect(self._onRename)
        actions.append(action_rename)

        self.setActions(actions)

        if keys:
            self.setDefaultAction(actions_open[keys[0]])

        # Defaults
        self.setIdentifier(identifier)

    def _onDelete(self):
        self.controller().datumDelete.emit(self.identifier())

    def _onRename(self):
        controller = self.controller()
        oldidentifier = self.identifier()

        dialog = _RenameDialog(oldidentifier, controller)
        dialog.setValidator(_DatumValidator(controller))
        if dialog.exec_():
            newidentifier = dialog.newIdentifier()
            controller.datumRename.emit(oldidentifier, newidentifier)

    def setIdentifier(self, identifier):
        self._identifier = identifier

        datum_class = type(self.controller().datum(identifier))
        text = _generate_description(identifier, datum_class)
        self.setText(0, text)

    def identifier(self):
        return self._identifier

class _DatumConditionsTreeItem(_ActionTreeItem):

    def __init__(self, identifier, controller, parent):
        _ActionTreeItem.__init__(self, controller, parent)
        self.setText(0, 'Conditions')

        # Actions
        action_add = QAction('Add condition', self.treeWidget())
        self.setActions([action_add])

        # Signals
        action_add.triggered.connect(self._onAdd)

        # Defaults
        self.setIdentifier(identifier)

    def _onAdd(self):
        parent = self.treeWidget()

        controller = self.controller()
        identifiers = controller.conditionIdentifiers() - \
            controller.datumConditionIdentifiers(self.identifier())

        if not identifiers:
            QMessageBox.critical(parent, 'Add condition',
                                 'All conditions are already added')
            return

        title = 'Add condition'
        message = 'Select a condition'
        items = list(identifiers)
        condition_identifier, answer = \
            QInputDialog.getItem(parent, title, message, items, editable=False)
        if answer:
            datum_identifier = self.identifier()
            self.controller().datumConditionAdd.emit(datum_identifier,
                                                     condition_identifier)

    def setIdentifier(self, identifier):
        self._identifier = identifier

    def identifier(self):
        return self._identifier

class _DatumConditionTreeItem(_ConditionTreeItem):

    def __init__(self, datum_identifier, condition_identifier, controller, parent):
        _ConditionTreeItem.__init__(self, condition_identifier, controller, parent)

        # Defaults
        self.setDatumIdentifier(datum_identifier)

    def _onDelete(self):
        self.controller().datumConditionDelete.emit(self.datumIdentifier(),
                                                    self.identifier())

    def setDatumIdentifier(self, identifier):
        self._datum_identifier = identifier

    def datumIdentifier(self):
        return self._datum_identifier

#--- Sub-windows

class _WidgetSubWindow(QMdiSubWindow):

    def __init__(self, controller, parent=None):
        QMdiSubWindow.__init__(self, parent)
        self.setAttribute(Qt.WA_DeleteOnClose)
        self.setWindowIcon(getIcon('text-x-generic'))

        # Variables
        self._controller = controller

        # Widgets
        scrollarea = QScrollArea()
        scrollarea.setWidgetResizable(True)

        # Layouts
        QMdiSubWindow.setWidget(self, scrollarea)

    def controller(self):
        return self._controller

    def widget(self):
        return QMdiSubWindow.widget(self).widget()

    def setWidget(self, widget):
        return QMdiSubWindow.widget(self).setWidget(widget)

class _HeaderSubWindow(_WidgetSubWindow):

    def __init__(self, controller, parent=None):
        _WidgetSubWindow.__init__(self, controller, parent)
        self.setWindowTitle('Header')

        # Widgets
        widget = HeaderWidget()
        header = self.controller().dataFileHeader()
        widget.setParameter(header)
        self.setWidget(widget)

        # Signals
        widget.edited.connect(self._onEdited)

    def _onEdited(self):
        try:
            header = self.widget().header()
        except:
            header = None
        self.controller().headerEdited.emit(header)

    def closeEvent(self, event):
        self.controller().headerClosed.emit()
        return _WidgetSubWindow.closeEvent(self, event)

class _ConditionSubWindow(_WidgetSubWindow):

    def __init__(self, identifier, controller, parent=None):
        _WidgetSubWindow.__init__(self, controller, parent)

        # Widgets
        condition = self.controller().condition(identifier)
        widget = self._initWidget(condition)
        widget.setParameter(condition)
        self.setWidget(widget)

        # Signals
        widget.edited.connect(self._onEdited)

        # Defaults
        self.setIdentifier(identifier)

    def _initWidget(self, condition):
        items = list(iter_condition_widgets(type(condition).__name__))
        if not items:
            raise Exception('Cannot display condition')
        if len(items) > 1:
            raise Exception('Too many displays associated with condition')
        return items[0][1]

    def _onEdited(self):
        try:
            condition = self.widget().condition()
        except:
            condition = None
        self.controller().conditionEdited.emit(self.identifier(), condition)

    def closeEvent(self, event):
        self.controller().conditionClosed.emit(self.identifier())
        return _WidgetSubWindow.closeEvent(self, event)

    def setIdentifier(self, identifier):
        self._identifier = identifier

        condition_class = type(self.controller().condition(identifier))
        title = _generate_description(identifier, condition_class)
        self.setWindowTitle(title)

    def identifier(self):
        return self._identifier

class _DatumSubWindow(_WidgetSubWindow):

    def __init__(self, identifier, modifier, controller, parent=None):
        _WidgetSubWindow.__init__(self, controller, parent)

        # Variables
        self._modifier = modifier

        # Widgets
        datum = self.controller().datum(identifier)
        widget = self._initWidget(datum, modifier)
        self.setWidget(widget)

        # Defaults
        self.setIdentifier(identifier)

    def _initWidget(self, datum, modifier):
        name = type(datum).__name__ + '.' + modifier
        items = list(iter_datum_widgets(name, datum))
        if not items:
            raise Exception('Cannot display datum')
        if len(items) > 1:
            raise Exception('Too many displays associated with datum')
        return items[0][1]

    def closeEvent(self, event):
        self.controller().datumClosed.emit(self.identifier(), self._modifier)
        return _WidgetSubWindow.closeEvent(self, event)

    def setIdentifier(self, identifier):
        self._identifier = identifier

        datum_class = type(self.controller().datum(identifier))
        title = _generate_description(identifier, datum_class)
        self.setWindowTitle(title)

    def identifier(self):
        return self._identifier

#--- Main widgets

class _Area(QMdiArea):

    def __init__(self, controller, parent=None):
        QMdiArea.__init__(self, parent)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)

        # Variables
        self._controller = controller
        self._header_window = None
        self._condition_windows = {}
        self._datum_windows = {}

        # Signals
        self.controller().dataFileClosed.connect(self._onDataFileClosed)
        self.controller().headerOpen.connect(self._onHeaderOpen)
        self.controller().headerClosed.connect(self._onHeaderClosed)
        self.controller().conditionOpen.connect(self._onConditionOpen)
        self.controller().conditionClosed.connect(self._onConditionClosed)
        self.controller().conditionRenamed.connect(self._onConditionRenamed)
        self.controller().conditionDeleted.connect(self._onConditionDeleted)
        self.controller().datumOpen.connect(self._onDatumOpen)
        self.controller().datumClosed.connect(self._onDatumClosed)
        self.controller().datumRenamed.connect(self._onDatumRenamed)
        self.controller().datumDeleted.connect(self._onDatumDeleted)

    def _onDataFileClosed(self):
        self.closeAllSubWindows()

    def _onHeaderOpen(self):
        window = self._header_window
        if window is None:
            window = _HeaderSubWindow(self.controller())
            self._header_window = window

        self._showWindow(window)

        self.controller().headerOpened.emit()

    def _onHeaderClosed(self):
        self._header_window = None

    def _onConditionOpen(self, identifier):
        window = self._condition_windows.get(identifier)
        if window is None:
            window = _ConditionSubWindow(identifier, self.controller())
            self._condition_windows[identifier] = window

        self._showWindow(window)

        self.controller().conditionOpened.emit(identifier)

    def _onConditionClosed(self, identifier):
        self._condition_windows.pop(identifier)

    def _onConditionRenamed(self, oldidentifier, newidentifier):
        if oldidentifier not in self._condition_windows:
            return
        window = self._condition_windows.pop(oldidentifier)
        window.setIdentifier(newidentifier)
        self._condition_windows[newidentifier] = window

    def _onConditionDeleted(self, identifier):
        window = self._condition_windows.get(identifier)
        if window is not None:
            window.close()

    def _onDatumOpen(self, identifier, modifier):
        window = self._datum_windows.get(identifier, {}).get(modifier)
        if window is None:
            window = _DatumSubWindow(identifier, modifier, self.controller())
            self._datum_windows.setdefault(identifier, {})[modifier] = window

        self._showWindow(window)

        self.controller().datumOpened.emit(identifier, modifier)

    def _onDatumClosed(self, identifier, modifier):
        self._datum_windows[identifier].pop(modifier)

    def _onDatumRenamed(self, oldidentifier, newidentifier):
        if oldidentifier not in self._datum_windows:
            return
        for window in self._datum_windows[oldidentifier].values():
            window.setIdentifier(newidentifier)

        oldmodifiers = self._datum_windows.pop(oldidentifier)
        self._datum_windows[newidentifier] = oldmodifiers

    def _onDatumDeleted(self, identifier):
        for window in list(self._datum_windows.get(identifier, {}).values()):
            window.close()

    def _showWindow(self, window):
        if window in self.subWindowList():
            self.setActiveSubWindow(window)
        else:
            self.addSubWindow(window)
        window.showNormal()
        window.raise_()

    def controller(self):
        return self._controller

class _Tree(QTreeWidget):

    def __init__(self, controller, parent=None):
        QTreeWidget.__init__(self, parent)
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.header().close()

        # Variables
        self._controller = controller
        self._header_item = None
        self._conditions_item = None
        self._condition_items = {}
        self._data_item = None
        self._datum_items = {}
        self._datum_conditions_item = {}

        # Signals
        self.customContextMenuRequested.connect(self._onContextMenu)
        self.itemDoubleClicked.connect(self._onDoubleClicked)

        self.controller().dataFileOpened.connect(self._onDataFileOpened)
        self.controller().dataFileClosed.connect(self._onDataFileClosed)
        self.controller().headerOpened.connect(self._onHeaderOpened)
        self.controller().headerClosed.connect(self._onHeaderClosed)
        self.controller().conditionOpened.connect(self._onConditionOpened)
        self.controller().conditionClosed.connect(self._onConditionClosed)
        self.controller().conditionAdded.connect(self._onConditionAdded)
        self.controller().conditionRenamed.connect(self._onConditionRenamed)
        self.controller().conditionDeleted.connect(self._onConditionDeleted)
        self.controller().datumOpened.connect(self._onDatumOpened)
        self.controller().datumClosed.connect(self._onDatumClosed)
        self.controller().datumRenamed.connect(self._onDatumRenamed)
        self.controller().datumDeleted.connect(self._onDatumDeleted)
        self.controller().datumConditionAdded.connect(self._onDatumConditionAdded)
        self.controller().datumConditionDeleted.connect(self._onDatumConditionDeleted)

    def _onDataFileOpened(self, datafile):
        c = self.controller()

        # Header
        item_header = _HeaderTreeItem(c, self)
        self._header_item = item_header

        # Conditions
        self._conditions_item = _ConditionsTreeItem(c, self)

        for identifier in sorted(datafile.conditions.keys()):
            item_condition = \
                _ConditionTreeItem(identifier, c, self._conditions_item)
            self._condition_items.setdefault(identifier, set()).add(item_condition)

        # Data
        self._data_item = _ActionTreeItem(c, self)
        self._data_item.setText(0, 'Data')

        for datum_identifier in sorted(datafile.data.keys()):
            item_datum = _DatumTreeItem(datum_identifier, c, self._data_item)
            self._datum_items[datum_identifier] = item_datum

            item_conditions = \
                _DatumConditionsTreeItem(datum_identifier, c, item_datum)
            self._datum_conditions_item[datum_identifier] = item_conditions

            conditions = datafile.data[datum_identifier].conditions
            for condition_identifier in sorted(conditions.keys()):
                item_condition = \
                    _DatumConditionTreeItem(datum_identifier, condition_identifier,
                                            c, item_conditions)
                self._condition_items.setdefault(condition_identifier, set()).add(item_condition)

        self.expandAll()

    def _onDataFileClosed(self):
        self.clear()
        self._header_item = None
        self._conditions_item = None
        self._condition_items.clear()
        self._data_item = None
        self._datum_items.clear()
        self._datum_conditions_item.clear()

    def _onHeaderOpened(self):
        item = self._header_item
        self._itemOpened(item)

    def _onHeaderClosed(self):
        item = self._header_item
        self._itemClosed(item)

    def _onConditionOpened(self, identifier):
        for item in self._condition_items[identifier]:
            self._itemOpened(item)

    def _onConditionClosed(self, identifier):
        for item in self._condition_items[identifier]:
            self._itemClosed(item)

    def _onDatumOpened(self, identifier, modifier):
        item = self._datum_items[identifier]
        self._itemOpened(item)

    def _onDatumClosed(self, identifier, modifier):
        item = self._datum_items[identifier]
        self._itemClosed(item)

    def _onConditionAdded(self, identifier, condition):
        item_conditions = self._conditions_item
        item = _ConditionTreeItem(identifier, self.controller(), item_conditions)
        self._condition_items.setdefault(identifier, set()).add(item)

    def _onConditionRenamed(self, oldidentifier, newidentifier):
        items = self._condition_items.pop(oldidentifier)
        for item in items:
            item.setIdentifier(newidentifier)
        self._condition_items[newidentifier] = items

    def _onConditionDeleted(self, identifier):
        for item in self._condition_items.pop(identifier):
            item.parent().removeChild(item)

    def _onDatumRenamed(self, oldidentifier, newidentifier):
        item = self._datum_items.pop(oldidentifier)
        item.setIdentifier(newidentifier)
        self._datum_items[newidentifier] = item

        item = self._datum_conditions_item.pop(oldidentifier)
        self._datum_conditions_item[newidentifier] = item

    def _onDatumDeleted(self, identifier):
        # Disconnect condition items
        item_conditions = self._datum_conditions_item[identifier]
        for item in item_conditions.takeChildren():
            self._condition_items[item.identifier()].remove(item)

        # Remove all items
        item = self._datum_items.pop(identifier)
        item.parent().removeChild(item)
        del self._datum_conditions_item[identifier]

    def _onDatumConditionAdded(self, datum_identifier, condition_identifier):
        item_conditions = self._datum_conditions_item[datum_identifier]

        item = _DatumConditionTreeItem(datum_identifier, condition_identifier,
                                       self.controller(), item_conditions)
        self._condition_items.setdefault(condition_identifier, set()).add(item)

    def _onDatumConditionDeleted(self, datum_identifier, condition_identifier):
        for item in self._condition_items[condition_identifier]:
            if isinstance(item, _DatumConditionTreeItem) and \
                    item.datumIdentifier() == datum_identifier:
                self._condition_items[condition_identifier].remove(item)
                item.parent().removeChild(item)
                break

    def _onContextMenu(self, point):
        item = self.itemAt(point)
        if item is None or not hasattr(item, 'popupMenu'):
            return

        menu = item.popupMenu()
        if menu is None or menu.isEmpty():
            return

        menu.exec_(self.mapToGlobal(point))

    def _onDoubleClicked(self, item, column):
        if not hasattr(item, 'popupMenu'):
            return

        menu = item.popupMenu()
        if menu is None or menu.isEmpty():
            return

        action = menu.defaultAction()
        if action is None:
            return

        action.trigger()

    def _itemOpened(self, item):
        font = item.font(0)
        font.setUnderline(True)
        item.setFont(0, font)

    def _itemClosed(self, item):
        font = item.font(0)
        font.setUnderline(False)
        item.setFont(0, font)

    def controller(self):
        return self._controller

class Viewer(QMainWindow):

    def __init__(self, parent):
        QMainWindow.__init__(self, parent)
        self.setWindowTitle("HMSA Viewer")
        self.setWindowIcon(getIcon('application-x-executable'))
        self.setAcceptDrops(True)

        # Variables
        self._controller = Controller()

        # Actions
        self._act_new = QAction(getIcon("document-new"), "&New", self)
        self._act_new.setShortcut(QKeySequence.New)
        self._act_open = QAction(getIcon("document-open"), "&Open", self)
        self._act_open.setShortcut(QKeySequence.Open)
        self._act_import = QAction("&Import", self)
        self._act_import.setShortcut(QKeySequence(Qt.CTRL + Qt.Key_I))
        self._act_import.setEnabled(bool(list(iter_importers())))
        self._act_close = QAction("&Close", self)
        self._act_close.setEnabled(False)
        self._act_revert = QAction(getIcon('document-revert'), '&Revert', self)
        self._act_revert.setEnabled(False)
        self._act_save = QAction(getIcon('document-save'), '&Save', self)
        self._act_save.setShortcut(QKeySequence.Save)
        self._act_save.setEnabled(False)
        self._act_saveas = QAction(getIcon('document-save-as'), 'Save as', self)
        self._act_saveas.setShortcut(QKeySequence.SaveAs)
        self._act_saveas.setEnabled(False)
        self._act_export = QAction('&Export', self)
        self._act_export.setShortcut(QKeySequence(Qt.CTRL + Qt.Key_E))
        self._act_export.setEnabled(False)
        self._act_preferences = QAction('Preferences', self)
        self._act_preferences.setShortcut(QKeySequence.Preferences)
        self._act_exit = QAction('Exit', self)
        self._act_exit.setShortcut(QKeySequence.Quit)

        self._act_window_cascade = QAction("Cascade", self)
        self._act_window_tile = QAction("Tile", self)
        self._act_window_closeall = QAction("Close all", self)

        self._act_about = QAction("About", self)

        # Widgets
        self._area = _Area(self._controller)
        self._tree = _Tree(self._controller)

        dck_datafile = QDockWidget("Explorer", self)
        dck_datafile.setAllowedAreas(Qt.LeftDockWidgetArea |
                                     Qt.RightDockWidgetArea)
        dck_datafile.setFeatures(QDockWidget.NoDockWidgetFeatures |
                                 QDockWidget.DockWidgetMovable)
        dck_datafile.setMinimumWidth(200)
        dck_datafile.setWidget(self._tree)

        self._dlg_progress = QProgressDialog()
        self._dlg_progress.setRange(0, 100)
        self._dlg_progress.setModal(True)
        self._dlg_progress.hide()

        # Menu
        mnu_file = self.menuBar().addMenu("&File")
        mnu_file.addAction(self._act_new)
        mnu_file.addAction(self._act_open)
        mnu_file.addAction(self._act_import)
        mnu_file.addSeparator()
        mnu_file.addAction(self._act_close)
        mnu_file.addSeparator()
        mnu_file.addAction(self._act_save)
        mnu_file.addAction(self._act_saveas)
        mnu_file.addAction(self._act_revert)
        mnu_file.addAction(self._act_export)
        mnu_file.addSeparator()
        mnu_file.addAction(self._act_preferences)
        mnu_file.addSeparator()
        mnu_file.addAction(self._act_exit)

        mnu_windows = self.menuBar().addMenu("Window")
        mnu_windows.addAction(self._act_window_cascade)
        mnu_windows.addAction(self._act_window_tile)
        mnu_windows.addSeparator()
        mnu_windows.addAction(self._act_window_closeall)

        mnu_help = self.menuBar().addMenu("Help")
        mnu_help.addAction(self._act_about)

        # Toolbar
        tbl_file = self.addToolBar("File")
        tbl_file.setMovable(False)
        tbl_file.addAction(self._act_new)
        tbl_file.addAction(self._act_open)
        tbl_file.addSeparator()
        tbl_file.addAction(self._act_save)
        tbl_file.addAction(self._act_saveas)
        tbl_file.addSeparator()
        tbl_file.addAction(self._act_revert)

        # Layouts
        self.setCentralWidget(self._area)
        self.addDockWidget(Qt.LeftDockWidgetArea, dck_datafile)

        # Signals
        self._act_new.triggered.connect(self._onNew)
        self._act_open.triggered.connect(self._onOpen)
        self._act_close.triggered.connect(self._onClose)
        self._act_import.triggered.connect(self._onImport)
        self._act_save.triggered.connect(self._onSave)
        self._act_saveas.triggered.connect(self._onSaveAs)
        self._act_revert.triggered.connect(self._onRevert)
        self._act_export.triggered.connect(self._onExport)
        self._act_preferences.triggered.connect(self._onPreferences)
        self._act_exit.triggered.connect(self._onExit)

        self._act_window_cascade.triggered.connect(self._onWindowCascade)
        self._act_window_tile.triggered.connect(self._onWindowTile)
        self._act_window_closeall.triggered.connect(self._onWindowCloseall)

        self._act_about.triggered.connect(self._onAbout)

        self.controller().dataFileOpened.connect(self._onDataFileOpened)
        self.controller().dataFileClosed.connect(self._onDataFileClosed)
        self.controller().dataFileEdited.connect(self._onDataFileEdited)
        self.controller().dataFileSaved.connect(self._onDataFileSaved)
        self.controller().dataFileImported.connect(self._onDataFileImported)
        self.controller().dataFileExported.connect(self._onDataFileExported)

        self._dlg_progress.canceled.connect(self.controller().dataFileOpenCancel)
        self.controller().dataFileOpenProgress.connect(self._onDialogProgressProgress)
        self.controller().dataFileOpenCancel.connect(self._onDialogProgressCancel)
        self.controller().dataFileOpenException.connect(self._onDialogProgressException)

        self._dlg_progress.canceled.connect(self.controller().dataFileSaveCancel)
        self.controller().dataFileSaveProgress.connect(self._onDialogProgressProgress)
        self.controller().dataFileSaveCancel.connect(self._onDialogProgressCancel)
        self.controller().dataFileSaveException.connect(self._onDialogProgressException)

        self._dlg_progress.canceled.connect(self.controller().dataFileImportCancel)
        self.controller().dataFileImportProgress.connect(self._onDialogProgressProgress)
        self.controller().dataFileImportCancel.connect(self._onDialogProgressCancel)
        self.controller().dataFileImportException.connect(self._onDialogProgressException)

        self._dlg_progress.canceled.connect(self.controller().dataFileExportCancel)
        self.controller().dataFileExportProgress.connect(self._onDialogProgressProgress)
        self.controller().dataFileExportCancel.connect(self._onDialogProgressCancel)
        self.controller().dataFileExportException.connect(self._onDialogProgressException)

        # Defaults
        settings = self.controller().settings()
        self.move(settings.value("pos", self.pos()))
        self.resize(settings.value("size", self.size()))

    def _onNew(self):
        if not self._checkSave():
            return
        self.controller().dataFileNew.emit()

    def _onOpen(self):
        if not self._checkSave():
            return

        settings = self.controller().settings()
        curdir = settings.value("opendir", os.getcwd())
        filepath, answer = \
            QFileDialog.getOpenFileName(self, "Open", curdir,
                                        "HMSA file format (*.xml *.hmsa)")

        if not answer or not filepath:
            return
        settings.setValue("opendir", os.path.dirname(filepath))

        self._dlg_progress.reset()
        self._dlg_progress.show()
        self.controller().dataFileOpen.emit(filepath)

    def _onImport(self):
        if not self._checkSave():
            return

        # Open file dialog
        settings = self.controller().settings()
        curdir = settings.value("importdir", settings.value("opendir", os.getcwd()))
        search_extra = settings.valueBool("search_extra", True)

        importers = {}
        fileformats = []
        exts = set()
        for name, ext, importer in iter_importers(search_extra=search_extra):
            exts.add('*' + ext)
            importers.setdefault(ext, []).append((name, importer))
            fileformats.append('{0} files [{1}] (*{1})'.format(name, ext))
        fileformats.sort()
        fileformats.append('All files (%s)' % (' '.join(exts),))

        filepath, answer = \
            QFileDialog.getOpenFileName(self, "Import", curdir,
                                        ';;'.join(fileformats), fileformats[-1])

        if not answer or not filepath:
            return
        settings.setValue("importdir", os.path.dirname(filepath))

        # Find importer
        ext = os.path.splitext(filepath)[1]
        possibilities = list(filter(lambda v: v[1].can_import(filepath),
                                    importers.get(ext, [])))
        if not possibilities:
            message = 'No possible importer for %s' % filepath
            QMessageBox.critical(self, 'Import', message)
            return
        elif len(possibilities) > 1:
            message = 'Select importer'
            items = dict(possibilities)
            name, answer = \
                QInputDialog.getItem(self, 'Import', message,
                                     list(items.keys()), editable=False)
            if not answer:
                return
            importer = items[name]
        else:
            importer = possibilities[0][1]
        logging.debug('Using importer: %s' % importer.__class__.__name__)

        # Import
        self._dlg_progress.reset()
        self._dlg_progress.show()
        self.controller().dataFileImport.emit(importer, filepath)

    def _onExport(self):
        if not self._isValid():
            return

        # Select exporter
        datafile = self.controller().dataFile()
        exporters = dict(filter(lambda v: v[1].can_export(datafile),
                                iter_exporters()))
        if not exporters:
            QMessageBox.critical(self, "Export",
                                 "No supported exporter for this data file")
            return

        identifier, answer = \
            QInputDialog.getItem(self, 'Export', 'Select exporter',
                                 list(exporters.keys()), editable=False)
        if not answer:
            return

        exporter = exporters[identifier]

        # Open file dialog
        settings = self.controller().settings()
        curdir = settings.value("exportdir", settings.value("opendir", os.getcwd()))

        dirpath = \
            QFileDialog.getExistingDirectory(self, 'Export directory', curdir)
        if not dirpath:
            return
        settings.setValue("exportdir", dirpath)

        # Import
        self._dlg_progress.reset()
        self._dlg_progress.show()
        self.controller().dataFileExport.emit(exporter, dirpath)

    def _onClose(self):
        if not self._checkSave():
            return
        self.controller().dataFileClosed.emit()

    def _onRevert(self):
        self.controller().dataFileRevert.emit()

    def _onSave(self):
        datafile = self.controller().dataFile()
        if datafile.filepath is None:
            return self._onSaveAs()

        if not self._isValid():
            return

        self._dlg_progress.reset()
        self._dlg_progress.show()
        self.controller().dataFileSave.emit(None)

    def _onSaveAs(self):
        if not self._isValid():
            return

        settings = self.controller().settings()
        curdir = settings.value("savedir", os.getcwd())

        filepath, answer = \
            QFileDialog.getSaveFileName(self, "Save as", curdir,
                                        "HMSA file format (*.xml *.hmsa)")

        if not answer or not filepath:
            return
        settings.setValue("savedir", os.path.dirname(filepath))

        if not os.path.splitext(filepath)[1]:
            filepath += '.hmsa'

        self._dlg_progress.reset()
        self._dlg_progress.show()
        self.controller().dataFileSave.emit(filepath)

    def _onPreferences(self):
        dialog = _PreferencesDialog(self.controller())
        dialog.exec_()

    def _onExit(self):
        self.close()

    def _onWindowCascade(self):
        self._area.cascadeSubWindows()

    def _onWindowTile(self):
        self._area.tileSubWindows()

    def _onWindowCloseall(self):
        self._area.closeAllSubWindows()

    def _onAbout(self):
        fields = {'version': __version__,
                  'copyright': __copyright__,
                  'license': __license__}
        message = 'pyHMSA Viewer (version {version})\n{copyright}\nLicensed under {license}'.format(**fields)
        QMessageBox.about(self, 'About', message)

    def _onDialogProgressProgress(self, progress, status):
        self._dlg_progress.setValue(progress * 100)
        self._dlg_progress.setLabelText(status)

    def _onDialogProgressCancel(self):
        self._dlg_progress.hide()

    def _onDialogProgressException(self, ex):
        self._dlg_progress.hide()
        messagebox.exception(self, ex)

    def _onDataFileOpened(self, datafile):
        self._dlg_progress.hide()

        self._act_saveas.setEnabled(True)
        self._act_close.setEnabled(True)
        self._act_revert.setEnabled(False)
        self._act_save.setEnabled(False)
        self._act_export.setEnabled(bool(list(iter_exporters())))

        self._updateTitle()

    def _onDataFileClosed(self):
        self._act_saveas.setEnabled(False)
        self._act_close.setEnabled(False)
        self._act_revert.setEnabled(False)
        self._act_save.setEnabled(False)
        self._act_export.setEnabled(False)
        self._updateTitle()

    def _onDataFileEdited(self):
        self._act_revert.setEnabled(True)
        self._act_save.setEnabled(True)
        self._updateTitle()

    def _onDataFileSaved(self, datafile):
        self._dlg_progress.hide()

        QMessageBox.information(self, 'HMSA Viewer', 'Saved')

        self._act_revert.setEnabled(False)
        self._act_save.setEnabled(False)

        self._updateTitle()

    def _onDataFileImported(self, datafile):
        self._dlg_progress.hide()

        self.controller().dataFileOpened.emit(datafile)
        self.controller().dataFileEdited.emit() # Edited by default

    def _onDataFileExported(self, filepaths):
        self._dlg_progress.hide()

        message = 'Data file exported to:\n'
        for filepath in filepaths:
            message += ' - %s\n' % filepath
        QMessageBox.information(self, 'HMSA Viewer', message)

    def _updateTitle(self):
        c = self.controller()

        title = 'HMSA Viewer'
        if c.hasDataFile():
            title += ' - '
            title += c.dataFile().filepath or 'Untitled'

        if c.isDataFileEdited():
            title += ' [Modified]'

        self.setWindowTitle(title)

    def _checkSave(self):
        if not self.controller().isDataFileEdited():
            return True

        message = 'Do you want to save the modifications?'
        answer = QMessageBox.question(self, 'Save modification', message,
                                      QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel)
        if answer == QMessageBox.Yes:
            self._act_save.trigger()
            return True
        elif answer == QMessageBox.No:
            return True
        else:
            return False

    def _isValid(self):
        return validate_widget(self._area)

    def closeEvent(self, event):
        if not self._checkSave():
            event.ignore()
        else:
            settings = self.controller().settings()
            settings.setValue("pos", self.pos())
            settings.setValue("size", self.size())

            event.accept()

    def dragEnterEvent(self, event):
        if not event.mimeData().hasFormat('text/plain'):
            event.ignore()
            return

        urls = event.mimeData().urls()
        if len(urls) > 1:
            event.ignore()
            return

        filepath = urls[0].toLocalFile()
        if not os.path.exists(filepath):
            event.ignore()
            return

        if os.path.splitext(filepath)[1] not in ['.xml', '.hmsa']:
            event.ignore()
            return

        event.acceptProposedAction()

    def dropEvent(self, event):
        filepath = event.mimeData().urls()[0].toLocalFile()

        if not self._checkSave():
            return

        self._dlg_progress.reset()
        self._dlg_progress.show()
        self.controller().dataFileOpen.emit(filepath)

    def controller(self):
        return self._controller

def _setup(argv):
    # Configuration directory
    dirpath = os.path.join(os.path.expanduser('~'), '.pyhmsa')
    if not os.path.exists(dirpath):
        os.makedirs(dirpath)

    # Logging
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG if '-d' in argv else logging.INFO)

    fmt = '%(asctime)s - %(levelname)s - %(module)s - %(lineno)d: %(message)s'
    formatter = logging.Formatter(fmt)

    handler = logging.FileHandler(os.path.join(dirpath, 'viewer.log'), 'w')
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    handler = logging.StreamHandler()
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    logging.info('Started pyHMSA viewer')
    logging.info('version = %s', __version__)
    logging.info('operating system = %s %s',
                 platform.system(), platform.release())
    logging.info('machine = %s', platform.machine())
    logging.info('processor = %s', platform.processor())

    # Catch all exceptions
    def _excepthook(exc_type, exc_obj, exc_tb):
        messagebox.exception(None, exc_obj)
        sys.__excepthook__(exc_type, exc_obj, exc_tb)
    sys.excepthook = _excepthook

def run():
    argv = sys.argv
    _setup(argv)
    app = QApplication(argv)
    dialog = Viewer(None)
    dialog.show()
    app.exec_()

if __name__ == '__main__':
    run()


