#!/usr/bin/env python
"""
================================================================================
:mod:`controller` -- Controller
================================================================================

.. module:: controller
   :synopsis: Controller

.. inheritance-diagram:: pyhmsa.gui.viewer.controller

"""

# Standard library modules.
import copy
import time

# Third party modules.
from PySide.QtCore import QObject, Signal, QThread

# Local modules.
from pyhmsa.datafile import DataFile
from pyhmsa.spec.header import Header
from pyhmsa.spec.condition.condition import _Condition
from pyhmsa.fileformat.datafile import DataFileReader, DataFileWriter
from pyhmsa.fileformat.importer.importer import _Importer
from pyhmsa.fileformat.exporter.exporter import _Exporter
from pyhmsa.gui.util.settings import Settings

# Globals and constants variables.

class _MonitorableWrapperThread(QThread):

    resultReady = Signal(object)
    progressUpdated = Signal(float, str)
    exceptionRaised = Signal(Exception)

    def __init__(self, monitorable, args=(), kwargs=None):
        QThread.__init__(self)
        self._monitorable = monitorable
        self._args = args
        if kwargs is None:
            kwargs = {}
        self._kwargs = kwargs
        self._is_cancelled = False

    def run(self):
        self._monitorable._start(*self._args, **self._kwargs)

        try:
            while self._monitorable.is_alive():
                self.progressUpdated.emit(self._monitorable.progress,
                                          self._monitorable.status)
                time.sleep(0.1)
        except Exception as ex:
            self.exceptionRaised.emit(ex)
            return
        finally:
            if not self._is_cancelled:
                self.progressUpdated.emit(self._monitorable.progress,
                                          self._monitorable.status)

        if not self._is_cancelled:
            try:
                result = self._monitorable.get()
            except Exception as ex:
                self.exceptionRaised.emit(ex)
            else:
                self.resultReady.emit(result)

    def cancel(self):
        self._is_cancelled = True
        self._monitorable.cancel()

class Controller(QObject):

    dataFileNew = Signal()
    dataFileOpen = Signal(str)
    dataFileOpenCancel = Signal()
    dataFileOpenProgress = Signal(float, str)
    dataFileOpenException = Signal(Exception)
    dataFileOpened = Signal(DataFile)
    dataFileClosed = Signal()
    dataFileEdited = Signal()
    dataFileSave = Signal(str)
    dataFileSaveCancel = Signal()
    dataFileSaveProgress = Signal(float, str)
    dataFileSaveException = Signal(Exception)
    dataFileSaved = Signal(DataFile)
    dataFileRevert = Signal()
    dataFileReverted = Signal()
    dataFileImport = Signal(_Importer, str)
    dataFileImportCancel = Signal()
    dataFileImportProgress = Signal(float, str)
    dataFileImportException = Signal(Exception)
    dataFileImported = Signal(DataFile)
    dataFileExport = Signal(_Exporter, str)
    dataFileExportCancel = Signal()
    dataFileExportProgress = Signal(float, str)
    dataFileExportException = Signal(Exception)
    dataFileExported = Signal(object)

    headerOpen = Signal()
    headerOpened = Signal()
    headerClosed = Signal()
    headerEdited = Signal(Header)

    conditionOpen = Signal(str)
    conditionOpened = Signal(str)
    conditionClosed = Signal(str)
    conditionAdd = Signal(str, _Condition)
    conditionAdded = Signal(str, _Condition)
    conditionDelete = Signal(str)
    conditionDeleted = Signal(str)
    conditionRename = Signal(str, str)
    conditionRenamed = Signal(str, str)
    conditionEdited = Signal(str, _Condition)

    datumOpen = Signal(str, str)
    datumOpened = Signal(str, str)
    datumClosed = Signal(str, str)
    datumDelete = Signal(str)
    datumDeleted = Signal(str)
    datumRename = Signal(str, str)
    datumRenamed = Signal(str, str)
    datumConditionAdd = Signal(str, str)
    datumConditionAdded = Signal(str, str)
    datumConditionDelete = Signal(str, str)
    datumConditionDeleted = Signal(str, str)

    def __init__(self):
        QObject.__init__(self) # Required to activate signals

        # Variables
        self._settings = Settings("HMSA", "Viewer")
        self._datafile = None
        self._base_datafile = None
        self._edited = False
        self._reader_thread = None
        self._writer_thread = None
        self._importer_thread = None
        self._exporter_thread = None

        # Signals
        self.dataFileNew.connect(self._onDataFileNew)
        self.dataFileOpen.connect(self._onDataFileOpen)
        self.dataFileOpenCancel.connect(self._onDataFileOpenCancel)
        self.dataFileOpenException.connect(self._onDataFileOpenException)
        self.dataFileOpened.connect(self._onDataFileOpened)
        self.dataFileClosed.connect(self._onDataFileClosed)
        self.dataFileEdited.connect(self._onDataFileEdited)
        self.dataFileSave.connect(self._onDataFileSave)
        self.dataFileSaveCancel.connect(self._onDataFileSaveCancel)
        self.dataFileSaveException.connect(self._onDataFileSaveException)
        self.dataFileSaved.connect(self._onDataFileSaved)
        self.dataFileRevert.connect(self._onDataFileRevert)
        self.dataFileImport.connect(self._onDataFileImport)
        self.dataFileImportCancel.connect(self._onDataFileImportCancel)
        self.dataFileImportException.connect(self._onDataFileImportException)
        self.dataFileExport.connect(self._onDataFileExport)
        self.dataFileExportCancel.connect(self._onDataFileExportCancel)
        self.dataFileExportException.connect(self._onDataFileExportException)

        self.headerEdited.connect(self._onHeaderEdited)
        self.headerEdited.connect(self.dataFileEdited)

        self.conditionAdd.connect(self._onConditionAdd)
        self.conditionDelete.connect(self._onConditionDelete)
        self.conditionRename.connect(self._onConditionRename)
        self.conditionEdited.connect(self._onConditionEdited)
        self.conditionAdded.connect(self.dataFileEdited)
        self.conditionDeleted.connect(self.dataFileEdited)
        self.conditionRenamed.connect(self.dataFileEdited)
        self.conditionEdited.connect(self.dataFileEdited)

        self.datumDelete.connect(self._onDatumDelete)
        self.datumRename.connect(self._onDatumRename)
        self.datumConditionAdd.connect(self._onDatumConditionAdd)
        self.datumConditionDelete.connect(self._onDatumConditionDelete)
        self.datumDeleted.connect(self.dataFileEdited)
        self.datumRenamed.connect(self.dataFileEdited)
        self.datumConditionAdded.connect(self.dataFileEdited)
        self.datumConditionDeleted.connect(self.dataFileEdited)

    def _onDataFileNew(self):
        datafile = DataFile()
        self.dataFileOpened.emit(datafile)

    def _onDataFileOpen(self, filepath):
        self._reader_thread = \
            _MonitorableWrapperThread(DataFileReader(), args=(filepath,))
        self._reader_thread.resultReady.connect(self.dataFileOpened)
        self._reader_thread.progressUpdated.connect(self.dataFileOpenProgress)
        self._reader_thread.exceptionRaised.connect(self.dataFileOpenException)
        self._reader_thread.start()

    def _onDataFileOpenCancel(self):
        if self._reader_thread is None:
            return
        self._reader_thread.cancel()
        self._reader_thread.quit()
        self._reader_thread.wait()

    def _onDataFileOpenException(self):
        self._reader_thread.quit()
        self._reader_thread.wait()

    def _onDataFileOpened(self, datafile):
        if self._datafile is not None:
            self.dataFileClosed.emit()

        self._datafile = datafile
        self._base_datafile = copy.deepcopy(datafile)
        self._edited = False

    def _onDataFileClosed(self):
        self._datafile = None
        self._base_datafile = None
        self._edited = False

    def _onDataFileEdited(self):
        self._edited = True

    def _onDataFileSave(self, filepath):
        if not filepath:
            filepath = None
        self._writer_thread = \
            _MonitorableWrapperThread(DataFileWriter(),
                                      args=(self._datafile, filepath))
        self._writer_thread.resultReady.connect(self.dataFileSaved)
        self._writer_thread.progressUpdated.connect(self.dataFileSaveProgress)
        self._writer_thread.exceptionRaised.connect(self.dataFileSaveException)
        self._writer_thread.start()

    def _onDataFileSaveCancel(self):
        if self._writer_thread is None:
            return
        self._writer_thread.cancel()
        self._writer_thread.quit()
        self._writer_thread.wait()

    def _onDataFileSaveException(self):
        self._writer_thread.quit()
        self._writer_thread.wait()

    def _onDataFileImport(self, importer, filepath):
        self._importer_thread = \
            _MonitorableWrapperThread(importer, args=(filepath,))
        self._importer_thread.resultReady.connect(self.dataFileImported)
        self._importer_thread.progressUpdated.connect(self.dataFileImportProgress)
        self._importer_thread.exceptionRaised.connect(self.dataFileImportException)
        self._importer_thread.start()

    def _onDataFileImportCancel(self):
        if self._importer_thread is None:
            return
        self._importer_thread.cancel()
        self._importer_thread.quit()
        self._importer_thread.wait()

    def _onDataFileImportException(self):
        self._importer_thread.quit()
        self._importer_thread.wait()

    def _onDataFileExport(self, exporter, dirpath):
        self._exporter_thread = \
            _MonitorableWrapperThread(exporter, args=(self._datafile, dirpath))
        self._exporter_thread.resultReady.connect(self.dataFileExported)
        self._exporter_thread.progressUpdated.connect(self.dataFileExportProgress)
        self._exporter_thread.exceptionRaised.connect(self.dataFileExportException)
        self._exporter_thread.start()

    def _onDataFileExportCancel(self):
        if self._exporter_thread is None:
            return
        self._exporter_thread.cancel()
        self._exporter_thread.quit()
        self._exporter_thread.wait()

    def _onDataFileExportException(self):
        self._exporter_thread.quit()
        self._exporter_thread.wait()

    def _onDataFileSaved(self, datafile):
        self._edited = False

    def _onDataFileRevert(self):
        datafile = self._base_datafile
        self.dataFileClosed.emit()
        self.dataFileOpened.emit(datafile)
        self.dataFileReverted.emit()

    def _onHeaderEdited(self, header):
        if header is None:
            return
        self._datafile.header.update(header)

    def _onConditionAdd(self, identifier, condition):
        if identifier in self._datafile.conditions:
            return
        self._datafile.conditions[identifier] = condition
        self.conditionAdded.emit(identifier, condition)

    def _onConditionEdited(self, identifier, condition):
        if condition is None:
            return
        self._datafile.conditions[identifier] = condition

    def _onConditionDelete(self, identifier):
        self._datafile.conditions.pop(identifier)
        self.conditionDeleted.emit(identifier)

    def _onConditionRename(self, oldidentifier, newidentifier):
        condition = self._datafile.conditions.pop(oldidentifier)
        self._datafile.conditions[newidentifier] = condition
        self.conditionRenamed.emit(oldidentifier, newidentifier)

    def _onDatumDelete(self, identifier):
        self._datafile.data.pop(identifier)
        self.datumDeleted.emit(identifier)

    def _onDatumRename(self, oldidentifier, newidentifier):
        datum = self._datafile.data.pop(oldidentifier)
        self._datafile.data[newidentifier] = datum
        self.datumRenamed.emit(oldidentifier, newidentifier)

    def _onDatumConditionAdd(self, datum_identifier, condition_identifier):
        condition = self._datafile.conditions[condition_identifier]
        datum = self._datafile.data[datum_identifier]
        datum.conditions[condition_identifier] = condition
        self.datumConditionAdded.emit(datum_identifier, condition_identifier)

    def _onDatumConditionDelete(self, datum_identifier, condition_identifier):
        datum = self._datafile.data[datum_identifier]
        datum.conditions.pop(condition_identifier)
        self.datumConditionDeleted.emit(datum_identifier, condition_identifier)

    def settings(self):
        return self._settings

    def isDataFileEdited(self):
        return self._edited

    def hasDataFile(self):
        return self._datafile is not None

    def dataFile(self):
        return self._datafile

    def dataFileHeader(self):
        return self._datafile.header

    def condition(self, identifier):
        return self._datafile.conditions[identifier]

    def hasCondition(self, identifier):
        return identifier in self._datafile.conditions

    def conditionIdentifiers(self):
        return frozenset(self._datafile.conditions.keys())

    def datum(self, identifier):
        return self._datafile.data[identifier]

    def hasDatum(self, identifier):
        return identifier in self._datafile.data

    def datumConditionIdentifiers(self, identifier):
        return frozenset(self._datafile.data[identifier].conditions.keys())




