#!/usr/bin/env python
"""
================================================================================
:mod:`datum` -- Base datum widgets
================================================================================

.. module:: datum
   :synopsis: Base datum widgets

.. inheritance-diagram:: pyhmsa.gui.spec.datum.datum

"""

# Standard library modules.

# Third party modules.
from PySide.QtGui import QWidget, QVBoxLayout, QTableView, QHeaderView, QSizePolicy
from PySide.QtCore import Qt

from matplotlib.backends.backend_qt4agg import \
    (FigureCanvasQTAgg as FigureCanvas,
     NavigationToolbar2QTAgg as NavigationToolbar)

# Local modules.

# Globals and constants variables.

class _DatumWidget(QWidget):

    def __init__(self, clasz, datum=None, parent=None):
        QWidget.__init__(self, parent)

        name = clasz.TEMPLATE
        if clasz.CLASS is not None:
            name += ' (%s)' % clasz.CLASS
        self.setAccessibleName(name)

        # Variables
        self._class = clasz

        # Layouts
        layout = QVBoxLayout()
        layout.addLayout(self._initUI()) # Initialize widgets
        self.setLayout(layout)

        # Defaults
        if datum is not None:
            self.setDatum(datum)

    def _initUI(self):
        return QVBoxLayout()

    def setDatum(self, datum):
        pass

    @property
    def CLASS(self):
        return self._class

class _DatumTableWidget(_DatumWidget):

    def __init__(self, clasz, datum=None, parent=None):
        _DatumWidget.__init__(self, clasz, datum, parent)

    def _initUI(self):
        # Widgets
        self._table = QTableView()
        self._table.horizontalHeader().setResizeMode(QHeaderView.Stretch)

        # Layouts
        layout = _DatumWidget._initUI(self)
        layout.addWidget(self._table)

        return layout

    def _createModel(self, datum):
        raise NotImplementedError

    def setDatum(self, datum):
        _DatumWidget.setDatum(self, datum)

        model = self._createModel(datum)
        self._table.setModel(model)

class _DatumFigureWidget(_DatumWidget):

    def __init__(self, clasz, datum=None, parent=None):
        _DatumWidget.__init__(self, clasz, datum, parent)

    def _initUI(self):
        # Widgets
        figure = self._createFigure()
        self._canvas = FigureCanvas(figure)
        self._canvas.setFocusPolicy(Qt.StrongFocus)
        self._canvas.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self._canvas.updateGeometry()

        self.toolbar = NavigationToolbar(self._canvas, self.parent())

        # Layouts
        layout = _DatumWidget._initUI(self)
        layout.addWidget(self._canvas)
        layout.addWidget(self.toolbar)

        return layout

    def _createFigure(self):
        raise NotImplementedError

    def _drawFigure(self, fig, datum):
        raise NotImplementedError

    def setDatum(self, datum):
        _DatumWidget.setDatum(self, datum)
        self._drawFigure(self._canvas.figure, datum)
        self._canvas.draw()
