#!/usr/bin/env python
"""
================================================================================
:mod:`analysislist` -- Analysis list widgets
================================================================================

.. module:: analysislist
   :synopsis: Analysis list widgets

.. inheritance-diagram:: pyhmsa.gui.spec.datum.analysislist

"""

# Standard library modules.

# Third party modules.
from PySide.QtGui import QSlider, QFormLayout
from PySide.QtCore import Qt

# Local modules.
from pyhmsa.gui.spec.datum.datum import _DatumWidget
from pyhmsa.gui.spec.datum.analysis import \
    (Analysis0DTableWidget, Analysis1DGraphWidget, Analysis1DTableWidget,
     Analysis2DGraphWidget, Analysis2DTableWidget)

from pyhmsa.spec.datum.analysislist import \
    AnalysisList0D, AnalysisList1D, AnalysisList2D

# Globals and constants variables.

class _AnalysisListWidget(_DatumWidget):

    def __init__(self, clasz, datum=None, parent=None):
        self._datum = None
        _DatumWidget.__init__(self, clasz, datum, parent)

    def _initUI(self):
        # Widgets
        self._slider = QSlider(Qt.Orientation.Horizontal)
        self._slider.setTickPosition(QSlider.TicksBelow)

        self._wdg_analysis = self._createAnalysisWidget()

        # Layouts
        layout = _DatumWidget._initUI(self)

        sublayout = QFormLayout()
        sublayout.addRow('Analyses', self._slider)
        layout.addLayout(sublayout)

        layout.addWidget(self._wdg_analysis)

        # Signals
        self._slider.valueChanged.connect(self._onSlide)

        return layout

    def _createAnalysisWidget(self):
        raise NotImplementedError

    def _onSlide(self, value):
        if self._datum is None:
            return
        analysis = self._datum.toanalysis(value)
        self._wdg_analysis.setDatum(analysis)

    def setDatum(self, datum):
        _DatumWidget.setDatum(self, datum)
        self._datum = datum
        self._slider.setMaximum(datum.analysis_count - 1)
        self._slider.setValue(0)
        self._onSlide(0)

class AnalysisList0DTableWidget(_AnalysisListWidget):

    def __init__(self, datum=None, parent=None):
        _AnalysisListWidget.__init__(self, AnalysisList0D, datum, parent)

    def _createAnalysisWidget(self):
        return Analysis0DTableWidget()

class AnalysisList1DTableWidget(_AnalysisListWidget):

    def __init__(self, datum=None, parent=None):
        _AnalysisListWidget.__init__(self, AnalysisList1D, datum, parent)

    def _createAnalysisWidget(self):
        return Analysis1DTableWidget()

class AnalysisList1DGraphWidget(_AnalysisListWidget):

    def __init__(self, datum=None, parent=None):
        _AnalysisListWidget.__init__(self, AnalysisList1D, datum, parent)

    def _createAnalysisWidget(self):
        return Analysis1DGraphWidget()

class AnalysisList2DTableWidget(_AnalysisListWidget):

    def __init__(self, datum=None, parent=None):
        _AnalysisListWidget.__init__(self, AnalysisList2D, datum, parent)

    def _createAnalysisWidget(self):
        return Analysis2DTableWidget()

class AnalysisList2DGraphWidget(_AnalysisListWidget):

    def __init__(self, datum=None, parent=None):
        _AnalysisListWidget.__init__(self, AnalysisList2D, datum, parent)

    def _createAnalysisWidget(self):
        return Analysis2DGraphWidget()
