import os
from logger import log, DEBUG

class ScriptGenerator(object):

    def __init__(self, environment_vars={}):
        self.env = environment_vars

    def script(self, language='sh'):
        header_lines = [
            '#!/bin/%s' % language,
            '# Auto-generated by projectenv'
        ]

        env_lines = []
        for k, v in self.env.iteritems():
            line = getattr(self, '_write_env_var_%s' % language)(k, v)
            env_lines.append(line)

        # We are relying on the fact that sorting is stable as of python>=2.3
        env_lines.sort(key=self._script_order_key)
        return '\n'.join(header_lines + env_lines)

    def write(self, file_name):
        for language in ['sh', 'csh']:
            self._write(file_name, language)

    def _write(self, file_name, language):
        file_name += '.' + language
        log('writing script', file_name)
        f = open(file_name, 'w+')
        if f:
            try:
                script = self.script(language)
                log(None, ' >> ' + script.replace('\n', '\n >> ') + '\n')
                if not DEBUG:
                    f.write(script + '\n')
            finally:
                f.close()

    def _write_env_var_sh(self, name, value):
        if value:
            return 'export %s=%s' % (name, value)
        else:
            return 'unset %s' % name

    def _write_env_var_csh(self, name, value):
        if value:
            return 'setenv %s %s' % (name, value)
        else:
            return 'unsetenv %s' % name

    def _script_order_key(self, line):
        """
        export/setenv commands must be in the correct order to prevent the
        possibility of restoring an environment variable to the wrong value

        """
        # the original environment vars should be saved first
        if (line.startswith('export _PROJECTENV_') or
        line.startswith('setenv _PROJECTENV_')):
            return 1
        # next the remaining environment vars should be set
        elif line.startswith('export') or line.startswith('setenv'):
            return 2
        # unsetting the prefixed environment variables should be done last
        elif line.startswith('unset _PROJECTENV_') or line.startswith('unsetenv _PROJECTENV_'):
            return 4
        # unsetting environment vars only happen once all of the environment
        # vars that should be set have been set
        else:
            return 3
