#!/usr/bin/env python
"""
Module _NAMEDTUPLE -- Alternate implementation of namedtuple
Sub-Package STDLIB of Package PLIB
Copyright (C) 2008-2012 by Peter A. Donis

Released under the GNU General Public License, Version 2
See the LICENSE and README files for more information

From the ActiveState recipe at

http://code.activestate.com/recipes/500261/

The only significant change is to remove the replace argument,
since the Python 2.6 version doesn't include it.
"""

from operator import itemgetter as _itemgetter
from keyword import iskeyword as _iskeyword
import sys as _sys


def namedtuple(typename, field_names, verbose=False):
    """Returns a new subclass of tuple with named fields.

    >>> Point = namedtuple('Point', 'x y')
    >>> Point.__doc__        # docstring for the new class
    'Point(x, y)'
    >>> p = Point(11, y=22)  # instantiate with positional args or keywords
    >>> p[0] + p[1]          # indexable like a plain tuple
    33
    >>> x, y = p             # unpack like a regular tuple
    >>> x, y
    (11, 22)
    >>> p.x + p.y            # fields also accessible by name
    33
    >>> d = p._asdict()      # convert to a dictionary
    >>> d['x']
    11
    >>> Point(**d)           # convert from a dictionary
    Point(x=11, y=22)
    >>> p._replace(x=100)    # like str.replace() but targets named fields
    Point(x=100, y=22)
    >>> 
    """
    
    # Parse and validate the field names. Validation serves two purposes,
    # generating informative error messages and preventing template
    # injection attacks.
    
    if isinstance(field_names, basestring):
        # Names separated by whitespace and/or commas
        field_names = field_names.replace(',', ' ').split()
    field_names = tuple(map(str, field_names))
    for name in (typename,) + field_names:
        if not min(c.isalnum() or c=='_' for c in name):
            raise ValueError(
                'Type and field names must be valid identifiers: %r' % name)
        if _iskeyword(name):
            raise ValueError(
                'Type and field names cannot be keywords: %r' % name)
        if name[0].isdigit():
            raise ValueError(
                'Type and field names must be valid identifiers: %r' % name)
    seen_names = set()
    for name in field_names:
        if name.startswith('_'):
            raise ValueError(
                'Field names cannot start with an underscore: %r' % name)
        if name in seen_names:
            raise ValueError(
                'Encountered duplicate field name: %r' % name)
        seen_names.add(name)
    
    # Create and fill-in the class template
    numfields = len(field_names)
    # Tuple repr without parens or quotes
    argtxt = repr(field_names).replace("'", "")[1:-1]
    reprtxt = ', '.join('%s=%%r' % name for name in field_names)
    template = '''class %(typename)s(tuple):
        '%(typename)s(%(argtxt)s)' \n
        __slots__ = () \n
        _fields = %(field_names)r \n
        def __new__(_cls, %(argtxt)s):
            return _tuple.__new__(_cls, (%(argtxt)s)) \n
        @classmethod
        def _make(cls, iterable, new=tuple.__new__, len=len):
            'Make a new %(typename)s object from a sequence or iterable'
            result = new(cls, iterable)
            if len(result) != %(numfields)d:
                raise TypeError('Expected %(numfields)d arguments, got %%d' %% len(result))
            return result \n
        def __repr__(self):
            return '%(typename)s(%(reprtxt)s)' %% self \n
        def _asdict(self):
            'Return a new dict which maps field names to their values'
            return dict(zip(self._fields, self)) \n
        def _replace(_self, **kwds):
            'Return a new %(typename)s object replacing specified fields with new values'
            result = _self._make(map(kwds.pop, %(field_names)r, _self))
            if kwds:
                raise ValueError('Got unexpected field names: %%r' %% kwds.keys())
            return result \n
        def __getnewargs__(self):
            return tuple(self) \n\n''' % locals()
    for i, name in enumerate(field_names):
        template += '        %s = _property(_itemgetter(%d))\n' % (name, i)
    if verbose:
        print template
    
    # Execute the template string in a temporary namespace
    namespace = dict(_itemgetter=_itemgetter,
                     __name__='namedtuple_%s' % typename,
                     _property=property, _tuple=tuple)
    try:
        exec template in namespace
    except SyntaxError, e:
        raise SyntaxError(e.message + ':\n' + template)
    result = namespace[typename]
    
    # For pickling to work, the __module__ variable needs to be set to the
    # frame where the named tuple is created.  Bypass this step in enviroments
    # where sys._getframe is not defined (Jython for example) or sys._getframe
    # is not defined for arguments greater than 0 (IronPython).
    try:
        result.__module__ = _sys._getframe(1).f_globals.get(
            '__name__', '__main__')
    except (AttributeError, ValueError):
        pass
    
    return result


if __name__ == '__main__':
    # Determine whether to print out the template string
    # on the first invocation, and the output from the
    # other demonstrations. Note that this is separate
    # from making doctest give verbose output.
    from sys import argv as _argv
    verbose = ('-p' in _argv)
    
    # verify that namedtuple instances can be pickled
    from cPickle import loads, dumps
    Point = namedtuple('Point', 'x, y', verbose=verbose)
    p = Point(x=10, y=20)
    assert p == loads(dumps(p, -1))
    
    # test and demonstrate ability to override methods
    class Point(namedtuple('Point', 'x y')):
        @property
        def hypot(self):
            return (self.x ** 2 + self.y ** 2) ** 0.5
        def __str__(self):
            return 'Point: x=%6.3f y=%6.3f hypot=%6.3f' % (
                self.x, self.y, self.hypot)
    
    for p in Point(3,4), Point(14,5), Point(9./7,6):
        if verbose:
            print p
    
    class Point(namedtuple('Point', 'x y')):
        'Point class with optimized _make() and _replace() without error-checking'
        _make = classmethod(tuple.__new__)
        def _replace(self, _map=map, **kwds):
            return self._make(_map(kwds.get, ('x', 'y'), self))
    
    _output = Point(11, 22)._replace(x=100)
    if verbose:
        print _output
    
    import doctest
    results = doctest.testmod()
    if verbose:
        TestResults = namedtuple('TestResults', 'failed attempted')
        print TestResults(*results)
