#!/usr/bin/env python
"""
PYIDSERVER.PY
Copyright (C) 2008 by Peter A. Donis

Python implementation of IDServer, a
command-line tool to query an internet
server and return information about it.
"""

import sys
import socket

from plib.utils import options

PROTO_DEFAULT = 'http'
BUFSIZE = 4096

def do_output(fileobj, s):
    fileobj.write(s)
    fileobj.flush()

quitmsgs = [None, "QUIT\r\n"]

protocols = {
    'ftp': (21, [None]),
    'http': (80, ["HEAD / HTTP/1.0\r\n\r\n"]),
    'imap': (143, [None, "A1 CAPABILITY\r\n", "A2 LOGOUT\r\n"]),
    'news': (119, quitmsgs),
    'pop': (110, quitmsgs),
    'smtp': (25, quitmsgs) }

def run_idserver(fileobj, arg, dns_only, protocol, portnum):
    """
    Query server and write results to fileobj.
    
    Server argument 'arg' should be a URL string, either an IP address
    or a host name. Currently there is no support for parsing the
    protocol and port number out of the URL; they must be supplied as
    separate parameters to this function. (That is, currently there is
    no support for URL's like http://somesite.com:80 .)
    
    If dns_only is true, only a DNS lookup is done; no connection is
    actually made to the server.
    
    The protocol should be one of the strings listed as keys in the
    protocols dictionary above.
    
    The port number should be an integer giving the port number on
    the server. (This parameter should only need to be used very
    rarely; almost always the port number is determined by the
    protocol as shown in the dictionary above.)
    
    The output file object 'fileobj' can be any file-like object
    that has a write and a flush method. See pyidserver-gui.py for
    an example of a file-like object that writes to a text control.
    """
    
    if dns_only:
        do_output(fileobj, "Doing DNS lookup on %s ...\n" % arg)
    else:
        proto_msg = port_msg = ""
        if protocol == "":
            protocol = PROTO_DEFAULT
        else:
            protocol = protocol.lower()
            proto_msg = " using %s" % protocol
        if protocol in protocols:
            proto_port, proto_items = protocols[protocol]
            if portnum == 0:
                portnum = proto_port
            else:
                port_msg = " on port %i" % portnum
        else:
            raise ValueError, "Invalid protocol: %s.\n" % protocol
    
    ipaddr = socket.gethostbyname(arg)
    if ipaddr == arg:
        # URL was an IP address, reverse lookup
        url = socket.gethostbyaddr(ipaddr)[0]
        do_output(fileobj, "Domain name for %s is %s.\n" % (ipaddr, url))
    else:
        # URL was a domain name, normal lookup
        url = arg
        do_output(fileobj, "IP address for %s is %s.\n" % (url, ipaddr))
    
    if not dns_only:
        do_output(fileobj, "Contacting %s%s%s ...\n" % (arg, proto_msg, port_msg))
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.connect((url, portnum))
        do_output(fileobj, "Connected ...\n")
        replied = False
        for item in proto_items:
            if item is not None:
                sock.sendall(item)
            if not replied:
                do_output(fileobj, "Server returned the following:\n")
                replied = True
            do_output(fileobj, sock.recv(BUFSIZE))
        sock.close()
        do_output(fileobj, "Connection closed.\n")

def run_main(outfile, arg, errfile=None, dns_only=False, protocol="", portnum=0):
    if errfile is None:
        errfile = outfile
    try:
        run_idserver(outfile, arg, dns_only, protocol, portnum)
    except ValueError:
        errfile.write(str(sys.exc_info()[1]))
    except (socket.error, socket.herror, socket.gaierror, socket.timeout):
        exc_type, exc_value, _ = sys.exc_info()
        errfile.write("%s %s\n" % tuple(map(str, (exc_type, exc_value))))

if __name__ == '__main__':
    _, def_dns, def_proto, def_port = run_main.func_defaults
    optlist = (
        ("-l", "--lookup", { 'action': "store_true",
            'dest': "dns_only", 'default': def_dns,
            'help': "Only do DNS lookup, no server query" } ),
        ("-p", "--protocol", { 'action': "store", 'type': "string",
            'dest': "protocol", 'default': def_proto,
            'help': "Use the specified protocol to contact the server" } ),
        ("-r", "--port", { 'action': "store", 'type': "int",
            'dest': "portnum", 'default': def_port,
            'help': "Use the specified port number to contact the server" } )
        )
    arglist = ["URL"]
    
    opts, args = options.parse_options(optlist, arglist)
    run_main(sys.stdout, args[0], sys.stderr, opts.dns_only, opts.protocol, opts.portnum)
