#!/usr/bin/env python
"""
Module POSTINSTALL -- Post-install script utilities
Sub-Package STDLIB of Package PLIB
Copyright (C) 2008-2013 by Peter A. Donis

Released under the GNU General Public License, Version 2
See the LICENSE and README files for more information

This module contains common utility routines for the
PLIB post-install scripts.
"""

import sys
import os
import compileall
import glob
import importlib
import sysconfig

from plib.stdlib.builtins import first


def write_setup_file(modname, module_vars, outpath, outfilename, startlines):
    """Write file generated by post-install script.
    """
    
    fullpath = os.path.join(outpath, outfilename)
    thismod = sys.modules[modname]
    vars = [(varname, getattr(thismod, varname)) for varname in module_vars]
    for varname, value in vars:
        print "{}: {}".format(varname, value)
    
    print "Writing module {}...".format(fullpath)
    outfile = open(fullpath, 'w')
    outfile.writelines(startlines + [
        '{} = {}{}'.format(varname, repr(value), os.linesep) for varname, value in vars
    ])
    outfile.close()
    
    print "Byte-compiling {}...".format(fullpath)
    compileall.compile_file(fullpath)


def get_install_scheme(import_name):
    """Find the installation scheme for module or package ``import_name``.
    
    This allows appropriate directories to be determined for post-install scripts
    that write or symlink additional files.
    """
    
    mod = importlib.import_module(import_name)
    dirpath = os.path.dirname(mod.__file__)
    if hasattr(mod, '__path__'):
        # Back out of the package directories
        for _ in xrange(len(import_name.split('.'))):
            dirpath = os.path.dirname(dirpath)
    paths = [
        (scheme, sysconfig.get_path(key, scheme))
        for key in ('purelib', 'platlib')
        for scheme in sysconfig.get_scheme_names()
    ]
    return first(
        scheme for scheme, path in paths
        if (path == dirpath) and scheme.startswith(os.name)
    )


def get_bin_dir(import_name):
    """Return the corresponding ``bin`` directory for module or package ``import_name``.
    """
    return sysconfig.get_path('scripts', get_install_scheme(import_name))


def get_share_dir(name, import_name=None):
    """Return the corresponding ``share`` directory for module or package ``name``.
    
    The ``import_name`` parameter allows for modules or packages which are
    imported under a different name than the name they are given in your
    ``setup.py`` file.
    """
    return os.path.join(
        sysconfig.get_path('data', get_install_scheme(import_name or name)),
        'share',
        name
    )


if os.name == 'posix':
    import compileall
    
    
    def _pre_output(binpath, examplepath):
        print "Creating symlinks in", binpath, "to examples in", examplepath, "..."
    
    
    def _post_output(examplepath):
        print "Byte-compiling examples..."
        compileall.compile_dir(examplepath)
    
    
    _glob_pattern = '*.py'
    _process_fn = os.symlink


elif os.name == 'nt':
    import shutil
    
    
    def _pre_output(binpath, examplepath):
        print "Copying examples from", examplepath, "to", binpath, "..."
    
    
    def _post_output(examplepath):
        pass
    
    
    _glob_pattern = '*.*'
    _process_fn = shutil.copyfile


def _process_file(srcfile, destfile):
    print(srcfile, destfile)
    if os.path.exists(destfile):
        os.remove(destfile)
    _process_fn(srcfile, destfile)


def setup_examples(name, import_name=None, remove_py=False):
    """Set up example programs for module or package ``name``.
    
    Assumes that the example programs will be installed in the
    ``examples`` subdirectory of the ``share`` directory for ``name``.
    (This will automatically happen if there is an ``examples``
    subdirectory of the source distribution that is marked as a
    data directory in the setup script. The ``plib`` packages that
    have example programs do this.) Installs symlinks (on POSIX
    systems) in the ``bin`` directory for ``name``, or copies the
    example programs to that directory (on Windows).
    
    The ``import_name`` parameter allows for cases where the name
    under which a module or package is imported is different from
    the name under which it is identified in ``setup.py``; for
    example, the Python 3 ``plib`` sub-packages are imported as
    ``plib.stdlib``, etc., but in ``setup.py`` they are named
    ``plib3.stdlib``, etc.
    
    The ``remove_py`` parameter removes the ``.py`` extension
    from example scripts when symlinking or copying; this allows
    the programs to be named more like standard executables for
    direct invocation, while still being importable as ``.py``
    files in their original example directories. (This actually
    becomes useful in cases like the ``pyidserver`` and ``scrips``
    examples, which are imported by GUI wrappers in the ``gui``
    package example programs.
    """
    
    binpath = get_bin_dir(import_name or name)
    sharepath = get_share_dir(name, import_name or name)
    examplepath = os.path.join(sharepath, "examples")
    
    _pre_output(binpath, examplepath)
    
    pyfiles = [
        filename for dirname in glob.glob(os.path.join(examplepath, '*'))
        for filename in glob.glob(os.path.join(dirname, _glob_pattern))
    ]
    
    for pyfile in pyfiles:
        basename = os.path.basename(pyfile)
        if remove_py:
            fname, ext = os.path.splitext(basename)
            if ext == ".py":
                basename = fname
        destfile = os.path.join(binpath, basename)
        _process_file(pyfile, destfile)
    
    if (sys.platform == 'darwin') and binpath.startswith('/opt/local') and (binpath != '/opt/local/bin'):
        # With MacPorts we need additional symlinks in /opt/local/bin, both for
        # the example programs and for the post-install scripts themselves
        print "Creating additional symlinks in /opt/local/bin ..."
        post_install_scripts = glob.glob(os.path.join(binpath, 'plib-setup-*'))
        for pyfile in itertools.chain(pyfiles, post_install_scripts):
            srcfile = os.path.join(binpath, os.path.basename(pyfile))
            destfile = os.path.join('/opt/local/bin', os.path.basename(pyfile))
            process_file(srcfile, destfile)
    
    _post_output(examplepath)
    
    print "{} examples setup done!".format(name.upper())
