import re
from pycounters import reporters
import pycounters.utils.munin


class DjangoCountersMuninPlugin(pycounters.utils.munin.Plugin):
    def __init__(self, json_output_file=None, category=None, extra_config=[]):
        """

        :param json_output_file: the name of the json file outputted by PyCounters JSONFileReporter
        :param category: Munin category if needed.
        :param extra_config: extra munin plugins you want, based on any PyCounter you may have.
        """
        super(DjangoCountersMuninPlugin, self).__init__(json_output_file=json_output_file,
                                                        config=extra_config)
        self.category = category


    def auto_generate_config_from_json(self, exclude_views=[], include_views=[]):
        """
        automaticaly build a munin config from a json file outputted by django_counter
        :param exclude_views: a list of views to exclude
        :param include_views: a lift of views to include
        :return:
        """
        values = reporters.JSONFileReporter.safe_read(self.output_file)

        counters = filter(lambda x: not re.match("^__.*__$", x), values.keys())
        counters = sorted(counters)

        # now they counters are sorted, you can start by checking prefixes

        title_prefix = self.category + ": " if self.category else ""

        config = []

        active_view = None
        active_config = None
        for counter in counters:
            if not counter.startswith("v_"):
                continue # not generated by count_view

            view_name, counter_name = counter.split(".", 1)
            view_name = view_name[2:] # remove v_

            if exclude_views and view_name in exclude_views:
                continue

            if include_views and not view_name in include_views:
                continue

            if active_view is None or active_view != view_name:
                ## new counter found
                active_view = view_name
                active_config = {}
                config.append(active_config)
                active_config["id"] = "django_counters_"
                active_config["id"] += self.category + "_" + view_name if self.category else counter
                active_config["global"] = dict(category=self.category,
                                               title="%sAverage times for view %s" % (title_prefix, view_name),
                                               vlabel="time")

            if counter_name == "_rps":
                # request per second, another chart
                rps_config = {}
                config.append(rps_config)
                rps_config["id"] = "django_counters_"
                rps_config["id"] += self.category + "_" + counter if self.category else counter
                rps_config["global"] = dict(category=self.category,
                                            title="%sRequests per second for view %s" % (title_prefix, view_name),
                                            vlabel="rps")

                rps_config["data"] = [dict(counter=counter, label="Requests per sec", draw="LINE1")]

            elif counter_name == "_total":
                active_config["data"] = [dict(counter=counter, label="Total", draw="LINE1")]

            else:
                active_config["data"].append(
                    dict(counter=counter, label=counter_name, draw="AREASTACK")
                )

        return config

    def output_config(self, config):
        auto_config = self.auto_generate_config_from_json()
        merged_config = config + auto_config if config else auto_config
        super(DjangoCountersMuninPlugin, self).output_config(merged_config)

    def output_data(self, config):
        auto_config = self.auto_generate_config_from_json()
        merged_config = config + auto_config if config else auto_config
        super(DjangoCountersMuninPlugin, self).output_data(merged_config)
