from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.contrib.auth.models import User


class Customer(models.Model):
    def get_upload_path(self):
        return 'uploads/'+self.pk
    initials = models.CharField(_('initials'), max_length=20, blank=True)
    first_name = models.CharField(_('first name'), max_length=100, blank=True)
    email = models.EmailField(_('email'), null=True, blank=True, default=None)
    last_name_prefix = models.CharField(_('prefix last name'), max_length=50, blank=True)
    last_name = models.CharField(_('last name'), max_length=50, blank=True)
    CHOICES = [('m', _('male')), ('f', _('female'))]
    gender = models.CharField(_('gender'), max_length=1, choices=CHOICES, blank=True)
    phone_number = models.CharField(_('phone number'), max_length=50, blank=True)
    phone_number2 = models.CharField(_('phone number 2'), max_length=50, blank=True)
    address = models.CharField(_('address'), max_length=100, blank=True)
    postal_code = models.CharField(_('postal code'), max_length=20, blank=True)
    city = models.CharField(_('city'), max_length=50, blank=True)
    birthday = models.DateField(_('birthday'), null=True, blank=True)
    birth_place = models.CharField(_('birth place'), max_length=50, null=True, blank=True)
    identification = models.ImageField(_('identification'), upload_to=get_upload_path, blank=True)
    identification_number = models.CharField(_('identification number'), max_length=20, blank=True)
    nationality = models.CharField(_('nationality'), max_length=50, blank=True)
    comments = models.TextField(_('comments'), blank=True)
    source = models.CharField(_('source'), max_length=100, blank=True)
    relationships = models.ManyToManyField('self', through='Relationship', symmetrical=False,
                                           related_name='related_to', blank=True)

    def __unicode__(self):
        name = self.initials
        if self.first_name:
            name += ' ('+self.first_name+') '
        name += self.last_name_prefix+' '+self.last_name
        if len(name) > 1:
            return name
        return self.email

    class Meta:
        verbose_name = _('customer')
        verbose_name_plural = _('customers')
        ordering = ['last_name', 'first_name']


class Relationship(models.Model):
    name = models.CharField(_('name'), max_length=100, blank=True)
    from_customer = models.ForeignKey(Customer, related_name='from_customers')
    to_customer = models.ForeignKey(Customer, related_name='to_customers')


def upload_path_handler(instance, filename):
    return "user_{id}/{file}".format(id=instance.customer.id, file=filename)


class File(models.Model):
    customer = models.ForeignKey(Customer, null=False, blank=False,
                                 verbose_name=_('customer'), on_delete=models.CASCADE)
    name = models.CharField(_('name'), max_length=100, blank=True)
    comments = models.TextField(_('description'), blank=True)
    file = models.FileField(_('file'), upload_to=upload_path_handler, blank=False)

    def __unicode__(self):
        return self.file.name


class Log(models.Model):
    customer = models.ForeignKey(Customer, null=False, blank=False,
                                 verbose_name=_('customer'), on_delete=models.CASCADE)
    author = models.ForeignKey(User, null=True, blank=True)
    submitted = models.DateTimeField(_('submitted'), auto_now_add=True)
    message = models.TextField(_('comments'), blank=False)

    def __unicode__(self):
        if self.author:
            return self.author.first_name+' '+self.author.last_name+' - '+self.submitted.strftime('%d %b %Y %H:%M')
        return self.submitted.strftime('%d %b %Y %H:%M')

    class Meta:
        verbose_name = _('log')
        verbose_name_plural = _('logs')
        ordering = ['-submitted']


class Task(models.Model):
    customer = models.ForeignKey(Customer, null=True, blank=True,
                                 verbose_name=_('customer'), on_delete=models.CASCADE)
    author = models.ForeignKey(User, null=True, blank=True)
    created = models.DateTimeField(_('created'), null=True, blank=True, auto_now_add=True)
    edited = models.DateTimeField(_('edited'), auto_now=True)
    due_date = models.DateField(_('due date'), null=True, blank=True)
    due_time = models.TimeField(_('due time'), null=True, blank=True)
    task = models.TextField(_('task'), blank=False)

    def __unicode__(self):
        return self.task

    class Meta:
        verbose_name = _('task')
        verbose_name_plural = _('tasks')
        ordering = ['due_date', 'due_time']