from django.contrib import admin
from models import *
from django.utils.translation import ugettext_lazy as _
from django.core.urlresolvers import reverse
from mailchimp import utils
from forms import AddToMailingListForm
from django.http import HttpResponseRedirect
from django.template import RequestContext
from django.shortcuts import render_to_response
from mailchimp.chimpy.chimpy import ChimpyException


class LogInline(admin.StackedInline):
    model = Log
    extra = 0
    fk_name = 'customer'
    exclude = ['author']


class RelationShipInline(admin.TabularInline):
    model = Customer.relationships.through
    extra = 0
    fk_name = 'from_customer'


class FileInline(admin.TabularInline):
    model = File
    fk_name = 'customer'
    extra = 0


class TaskInline(admin.TabularInline):
    model = Task
    extra = 0
    fk_name = 'customer'
    exclude = ['author']


class TaskAdmin(admin.ModelAdmin):
    list_display = ['__unicode__', 'task', 'due_date', 'due_time', 'get_customer']

    def get_customer(self, obj):
        if obj.customer:
            return '<a href="%s">%s</a>' % (
                reverse('admin:crm_customer_change', args=(obj.customer.pk,)), obj.customer
            )
        return None

    get_customer.allow_tags = True
    get_customer.short_description = _('customer')

    exclude = ['author']


class CustomerAdmin(admin.ModelAdmin):
    list_display = ['__unicode__', 'email', 'phone_number', 'phone_number2', 'address', 'postal_code', 'city',
                    'nationality']
    list_filter = ['city', 'nationality', 'source']
    search_fields = ['initials', 'first_name', 'last_name', 'last_name_prefix',
                     'phone_number', 'phone_number2', 'city',
                     'nationality', 'email', 'address', 'postal_code',
                     'identification_number', 'comments',
                     'source']

    fieldsets = (
        (None, {
            'fields': ('initials', ('first_name', 'last_name_prefix', 'last_name'), 'gender',
                       ('birthday', 'birth_place'))
        }),
        (_('Identification'), {
            'fields': ('identification', 'identification_number', 'nationality')
        }),
        (_('Contact details'), {
            'fields': ('email', ('phone_number', 'phone_number2'))
        }),
        (_('Address'), {
            'fields': ('address', ('postal_code', 'city'))
        }),
        (_('Extra'), {
            'fields': ('comments', 'source')
        })
    )

    inlines = [
        RelationShipInline, LogInline, TaskInline, FileInline
    ]

    def save_formset(self, request, form, formset, change):
        instances = formset.save(commit=False)

        for instance in instances:
            if isinstance(instance, Log) or isinstance(instance, Task):
                if not instance.author:
                    instance.author = request.user
            instance.save()
        formset.save_m2m()

    def add_to_mailinglist(self, request, queryset):
        form = None

        if 'apply' in request.POST:
            form = AddToMailingListForm(request.POST)

            if form.is_valid():
                list_id = form.cleaned_data['list']
                list = utils.get_connection().get_list_by_id(list_id)

                count = 0
                false_count = 0
                for customer in queryset:
                    try:
                        list.subscribe(customer.email, {'EMAIL': customer.email,
                                                        'FNAME': customer.first_name,
                                                        'LNAME': customer.last_name_prefix+' '+customer.last_name},
                                       double_optin=False,
                                       update_existing=True)
                        count += 1
                    except ChimpyException:
                        false_count += 1

                plural = ''
                if count != 1:
                    plural = 's'

                self.message_user(request,
                                  "Successfully added %d/%d customer%s to mailchimp list %s." % (count,
                                                                                                 count+false_count,
                                                                                                 plural,
                                                                                                 list.name))
                return HttpResponseRedirect(request.get_full_path())

        if not form:
            form = AddToMailingListForm(initial={'_selected_action': request.POST.getlist(admin.ACTION_CHECKBOX_NAME)})

        return render_to_response('admin/add_to_list.html', {'customers': queryset,
                                                             'form': form}, context_instance=RequestContext(request))

    actions = ['add_to_mailinglist']


admin.site.register(Customer, CustomerAdmin)
admin.site.register(Task, TaskAdmin)
