#!/usr/bin/env python

import datetime
import re

from sqlalchemy import Column, MetaData, Table
from sqlalchemy import Boolean, Date, DateTime, Float, Integer, String, Time
from sqlalchemy.schema import CreateTable

DIALECTS = {
    'access': 'access.base',
    'firebird': 'firebird.kinterbasdb',
    'informix': 'informix.informixdb',
    'maxdb': 'maxdb.sapdb',
    'mssql': 'mssql.pyodbc',
    'mysql': 'mysql.mysqlconnector',
    'oracle': 'oracle.cx_oracle',
    'postgresql': 'postgresql.psycopg2',
    'sqlite': 'sqlite.pysqlite',
    'sybase': 'sybase.pyodbc'
}

NULL_COLUMN_MAX_LENGTH = 32

def make_column(column):
    """
    Creates a sqlalchemy column from a csvkit Column.
    """
    sql_column_kwargs = {'nullable': False}
    sql_type_kwargs = {}

    column_types = {
        bool: Boolean,
        int: Integer,
        float: Float,
        datetime.datetime: DateTime,
        datetime.date: Date,
        datetime.time: Time,
        None: String,
        unicode: String
        }

    max_lengths = {
        None: NULL_COLUMN_MAX_LENGTH,
        unicode: column.max_length
        }

    if column.type in column_types:
        sql_column_type = column_types[column.type]
    else:
        raise ValueError('Unexpected normalized column type: %s' % column.type)

    if column.type in max_lengths:
        sql_type_kwargs['length'] = max_lengths[column.type]

    sql_column_kwargs['nullable'] = column.nullable

    return Column(column.name, sql_column_type(**sql_type_kwargs), **sql_column_kwargs)

def make_table(csv_table, name='table_name'):
    """
    Creates a sqlalchemy table from a csvkit Table.
    """
    metadata = MetaData()
    sql_table = Table(csv_table.name, metadata)

    for column in csv_table:
        sql_table.append_column(make_column(column))

    return sql_table

def make_create_table_statement(sql_table, dialect=None):
    """
    Generates a CREATE TABLE statement for a sqlalchemy table.
    """
    if dialect:
        module = __import__('sqlalchemy.dialects.%s' % DIALECTS[dialect], fromlist=['dialect'])
        sql_dialect = module.dialect()
    else:
        sql_dialect = None 

    return unicode(CreateTable(sql_table).compile(dialect=sql_dialect)).strip() + ';'

def de_parameterize_insert_query(statement, values, dialect=None):
    """
    Insert row values in a parameterized insert statement generated by sqlalchemy.

    Note: all the hard-coded processing in this function sucks, but sqlalchemy doesn't expose a way of doing this as part of the dialect-specific query processing.
    """
    # All dialects use VALUES to set off the start of the parameter clause
    start_params = statement.index('VALUES (') + 7 
    params_clause = statement[start_params:]
    
    # Get an iterator for the values (which are likely a tuple)
    values = iter(values)

    # Use probably ill-advised closure magic to format values in way SQL understands
    def value_formatter(*args):
        v = values.next()

        if v is None:
            return 'NULL' 
        elif type(v) == bool:
            return unicode(v).upper()
        elif type(v) == unicode:
            return u'\'%s\'' % v 
        else:
            return unicode(v)

    # ':column_name' syntax
    if dialect in ['access', 'firebird', 'maxdb', 'mssql', 'oracle', 'sybase', None]:
        params_clause = re.sub('(:"?.+?"?)(?=[,)])', value_formatter, params_clause)
    # '?' syntax
    elif dialect in ['informix', 'sqlite']:
        params_clause = re.sub('(\?)(?=[,)])', value_formatter, params_clause)
    # '%s' syntax
    elif dialect in ['mysql']:
        params_clause = re.sub('(%s)(?=[,)])', value_formatter, params_clause)
    # '%(column_name)s' syntax
    elif dialect in ['postgresql']:
        params_clause = re.sub('(%("?.+?"?)s)(?=[,)])', value_formatter, params_clause)

    return statement[:start_params] + params_clause

def make_insert_statement(sql_table, values, dialect=None):
    """
    Generates INSERT statements for data in a Table using a sqlalchemy dialect.
    """
    if dialect:
        module = __import__('sqlalchemy.dialects.%s' % DIALECTS[dialect], fromlist=['dialect'])
        sql_dialect = module.dialect()
    else:
        sql_dialect = None

    insert = unicode(sql_table.insert().compile(dialect=sql_dialect))
    statement = de_parameterize_insert_query(insert, values, dialect=dialect)

    return statement.strip() + ';'

