#!/usr/bin/env python
"""Automate the final steps of configuring the CloudBioLinux exome example.

- Updates configuration file with server details.
- Provides links to custom Galaxy instance
- Install latest processing pipeline
- Start toplevel processing server to run exome analysis.
- Adds RabbitMQ user and virtual host with correct permissions.

Run the script with sudo or the root user.
"""
import os
import time
import shutil
import socket
import subprocess
import contextlib
import ConfigParser

import yaml

def main():
    config_dir = "/export/data/galaxy"
    work_dir = "/export/data/work"
    work_user = "galaxy"
    amqp_config = os.path.join(config_dir, "universe_wsgi.ini")
    pp_config = os.path.join(config_dir, "post_process.yaml")
    wait_until_mounted(amqp_config)
    update_amqp_config(amqp_config, socket.getfqdn())
    amqp_user, amqp_pass = read_ampq_config(amqp_config)
    amqp_vhost = read_pp_config(pp_config)
    install_latest_pipeline()
    setup_custom_galaxy()
    run_nextgen_analysis_server(pp_config, work_dir, work_user)
    setup_rabbitmq(amqp_vhost, amqp_user, amqp_pass)

def setup_custom_galaxy():
    """Provide links to custom Galaxy instance on shared volume.
    """
    galaxy_path = "/mnt/galaxyTools/galaxy-central"
    custom_galaxy_path = "/mnt/galaxyData/galaxy-central-hbc"
    storage_dir = "/export/data/upload/storage"
    if not os.path.exists(galaxy_path):
        subprocess.check_call(["mkdir", "-p", os.path.split(galaxy_path)[0]])
        subprocess.check_call(["ln", "-s", custom_galaxy_path, galaxy_path])
        subprocess.check_call(["chown", "-R", "galaxy:galaxy",
                               os.path.split(galaxy_path)[0]])
    subprocess.check_call(["chmod", "a+rwx", storage_dir])

def install_latest_pipeline():
    url = "git://github.com/chapmanb/bcbb.git"
    tmpdir = "/tmp"
    with chdir(tmpdir):
        subprocess.check_call(["git", "clone", url])
        with chdir(os.path.join(tmpdir, "bcbb", "nextgen")):
            subprocess.check_call(["python", "setup.py", "install"])
        shutil.rmtree("bcbb")

UPSTART_SCRIPT = """
description   "Nextgen sequencing analysis server"

start on runlevel [2345]

pre-start script
    mkdir -p {work_dir}
    chown -R {user} {work_dir}
end script

exec su -l -c 'nextgen_analysis_server.py -q toplevel {config_file} --basedir={work_dir}' {user}
"""


def run_nextgen_analysis_server(pp_config, work_dir, work_user):
    """Run a nextgen sequencing server using Ubuntu upstart.
    """
    upstart_file = "/etc/init/nextgen-analysis.conf"
    with open(upstart_file, "w") as out_handle:
        out_handle.write(UPSTART_SCRIPT.format(config_file=pp_config,
                                               work_dir=work_dir,
                                               user=work_user))
    subprocess.check_call(["service", "nextgen-analysis", "start"])

def setup_rabbitmq(vhost, user, passwd):
    """Add virtual host, user and password to RabbitMQ.
    """
    base_cl = ["rabbitmqctl"]
    subprocess.check_call(base_cl + ["add_user", user, passwd])
    subprocess.check_call(base_cl + ["add_vhost", vhost])
    subprocess.check_call(base_cl + ["set_permissions", "-p", vhost,
                                     user, '.*', '.*', '.*'])

def read_pp_config(fname):
    """Read AMQP vhost from YAML configuration file.
    """
    with open(fname) as in_handle:
        config = yaml.load(in_handle)
    return config["distributed"]["rabbitmq_vhost"]

def read_ampq_config(fname):
    """Get AMQP username and password from configuration file
    """
    config = ConfigParser.ConfigParser()
    config.read(fname)
    return (config.get("galaxy_amqp", "userid"),
            config.get("galaxy_amqp", "password"))

def update_amqp_config(fname, hostname):
    """Update AMQP configuration with internal hostname.
    """
    orig_stat = os.stat(fname)
    backup_file = "{0}.bak".format(fname)
    shutil.move(fname, backup_file)
    with open(backup_file) as in_handle:
        with open(fname, "w") as out_handle:
            in_amqp = False
            for line in in_handle:
                if line.startswith("[galaxy_amqp]"):
                    in_amqp = True
                if in_amqp and line.startswith("host ="):
                    line = "host = {0}\n".format(hostname)
                    in_amqp = False
                out_handle.write(line)
    # make updated file readable by initial user
    os.chown(fname, orig_stat.st_uid, orig_stat.st_gid)

# ## Utilities

def wait_until_mounted(fname):
    """Wait up to 3 minutes for mounted directory with file.
    """
    max_tries = 36
    wait_sec = 5
    num_tries = 0
    while 1:
        if os.path.exists(fname):
            break
        elif num_tries > max_tries:
            raise ValueError("Did not find {f} after {s} seconds.".format(
                f=fname, s=max_tries*wait_sec))
        time.sleep(wait_sec)
        num_tries += 1

@contextlib.contextmanager
def chdir(new_dir):
    """Context manager to temporarily change to a new directory.
    http://lucentbeing.com/blog/context-managers-and-the-with-statement-in-python/
    """
    cur_dir = os.getcwd()
    os.chdir(new_dir)
    try:
        yield
    finally:
        os.chdir(cur_dir)
 
if __name__ == "__main__":
    main()
