#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

'''LaTeX extension for Python-Markdown'''

import logger
import markdown
import md5
import os
import re
import subprocess
import sys
import tempfile
from cStringIO import StringIO

tex_document = r'\documentclass[%spt]{article}%s\pagestyle{empty}\begin{document}%s\end{document}'
math_packages = ('amsmath', 'amsfonts', 'amssymb')
inline_math = r'$%s$'
para_math = r'\begin{equation*}%s\end{equation*}'

blacklist = ('include', 'def', 'command', 'loop', 'repeat', 'open', 'toks', 'output', 'input', 'catcode', 'name', '^^', r'\every', r'\errhelp', r'\errorstopmode', r'\scrollmode', r'\nonstopmode', r'\batchmode', r'\read', r'\write', 'csname', r'\newhelp', r'\uppercase', r'\lowercase', r'\relax', r'\aftergroup', r'\afterassignment', r'\expandafter', r'\noexpand', r'\special')

class LaTeXPattern(markdown.inlinepatterns.Pattern):
    def __init__(self, pattern, markdown_instance, ext, ismath, packages=None):
        '''Initializes the Pattern object which will produce LaTeX output'''
        markdown.inlinepatterns.Pattern.__init__(self, pattern, markdown_instance)
        self.ext = ext
        self.ismath = ismath
        self.packages = []
        if self.ismath:
            self.packages.extend(math_packages)
        if packages:
            self.packages.extend(packages)
        package_string = ''.join(map(lambda s: r'\usepackage{%s}' % s, self.packages))
        font_size = self.ext.getConfig('font_size')
        self.tex_document = tex_document % (font_size, package_string, '%s')

    def handleMatch(self, m):
        '''Handles a match for the LaTeX pattern found by Markdown's processor.

        @param m the regular expression match for the line on which the pattern was
        found. m contains one capturing group before the part that matched this
        pattern and one after.'''
        req_logger = logger.get_logger()

        content = m.group(2).strip()
        if len(content) == 0:
            req_logger.notice('no content')
            return ''

        if self.ismath:
            if len(m.group(1).strip()) or len(m.group(3).strip()):
                latex_markup = inline_math % content
            else:
                latex_markup = para_math % content
        else:
            latex_markup = content

        # Create a unique filename from the snippet of LaTeX code
        hashcode = md5.new(latex_markup).hexdigest()
        req_logger.debug('mdx_latex rendering "%s", hashcode=%s' % (latex_markup, hashcode))

        # Alias configuration options
        image_dir = self.ext.getConfig('image_dir')
        image_url = self.ext.getConfig('image_url')
        background = self.ext.getConfig('background')

        # Define some callbacks for when the latex process is finished
        def success():
            img = markdown.etree.Element('img')
            img.set('src', image_url + '/%s.png' % hashcode)
            img.set('alt', content)
            img.set('class', 'latex')
            return img

        def error():
            err = markdown.etree.Element('span')
            err.set('class', 'message')
            err_text = markdown.etree.SubElement(err, 'tt')
            err_text.text = markdown.AtomicString('error rendering: ' + content.replace('\\', '\\\\'))
            return err

        # Check and see if this file has already been generated
        if os.path.exists(os.path.join(image_dir, hashcode + '.png')):
            req_logger.debug('image has already been generated')
            return success()

        # Check and see if there are any blacklisted LaTeX tags in the code
        for elem in blacklist:
            if latex_markup.find(elem) >= 0:
                req_logger.notice('blacklisted markup found: %s' % elem)
                return error()

        tmpdir = tempfile.mkdtemp()
        try:
            dvipath = os.path.join(tmpdir, 'ltmp.dvi')
            auxpath = os.path.join(tmpdir, 'ltmp.aux')
            logpath = os.path.join(tmpdir, 'ltmp.log')
            pngpath = os.path.join(image_dir, hashcode + '.png')

            # an undocumented and quite possibly unportable feature:
            # setting stdout = -1 and stderr = -1 discards stdout and stderr
            # equivalent to redirecting to /dev/null
            sp_args = ['latex', '-jobname=ltmp', '-interaction=batchmode', '-kpathsea-debug=8', self.tex_document % latex_markup]
            popen = subprocess.Popen(sp_args, cwd = tmpdir, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
            output, errput = popen.communicate()
            if popen.returncode != 0:
                req_logger.debug('latex command "%s" failed with\nstdout:%s\nstderr:%s' % (' '.join(sp_args), output, errput))
                req_logger.debug('log contents:\n')
                sys.stderr.write(open(logpath, 'r').read())
                return error()
            else:
                del output, errput, popen
            sp_args = ['dvipng', '-D', '96', '-o', pngpath, '-T', 'tight', '-bg', background, dvipath]
            popen = subprocess.Popen(sp_args, cwd = tmpdir, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
            output, errput = popen.communicate()
            if popen.returncode != 0:
                req_logger.debug('dvipng command "%s" failed with\nstdout:%s\nstderr:%s' % (' '.join(sp_args), output, errput))
                return error()
            else:
                del output, errput, popen
        finally:
            if os.path.isfile(auxpath):
                os.remove(auxpath)
            if os.path.isfile(dvipath):
                os.remove(dvipath)
            if os.path.isfile(logpath):
                os.remove(logpath)
            os.rmdir(tmpdir)
        req_logger.debug('returning successfully')
        return success()


class LaTeXExtension(markdown.Extension):
    '''The actual extension class.'''
    def __init__(self, configs):
        self.config = {'image_dir': ['.', 'The directory to hold generated images'],
                       'image_url': ['/images/latex', 'The URL prefix to generated images'],
                       'background': ['Transparent', 'The background color for the generated PNG image'],
                       'font_size': ['11', 'The font size in points'],
                       'packages': ['', 'Extra LaTeX packages to include']}
        req_logger = logger.get_logger()
        # Markdown should do this itself but it's stupid :-(
        for key, value in configs:
            if key in self.config:
                req_logger.debug('overriding config: %s=%s' % (key, value))
                self.config[key][0] = value

    def extendMarkdown(self, md, md_globals):
        packages = self.config['packages'][0].split()
        md.inlinePatterns.insert(0, 'latex', LaTeXPattern(r'\\begin\{latex\}(.*)\\end\{latex\}', md, self, False, packages))
        md.inlinePatterns.insert(0, 'latex_math', LaTeXPattern(r'(\\begin\{equation\*?\}.*\\end\{equation\*?\})', md, self, True, packages))
        md.inlinePatterns.insert(0, 'latex_inline', LaTeXPattern(r'\$(?=\S)([^$]+?)(?<=\S)\$', md, self, True, packages))

def makeExtension(configs):
    '''Returns an instance of the extension.'''
    return LaTeXExtension(configs)
