// Copyright 2014 MINES ParisTech
//
// This file is part of LinPy.
//
// LinPy is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// LinPy is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with LinPy.  If not, see <http://www.gnu.org/licenses/>.

#include <Python.h>

#include <isl/constraint.h>
#include <isl/set.h>
#include <isl/vertices.h>


static inline int _append_pointer_to_list(void *pointer, void *user) {
    PyObject *list, *item;
    list = (PyObject *) user;
    item = PyLong_FromVoidPtr(pointer);
    if (item == NULL) {
        return -1;
    }
    return PyList_Append(list, item);
}

static int _append_constraint_to_list(isl_constraint *constraint, void *user) {
    return _append_pointer_to_list((void *) constraint, user);
}

static PyObject * isl_basic_set_constraints(PyObject *self, PyObject *args) {
    long pointer;
    isl_basic_set *bset;
    PyObject *list;
    if (!PyArg_ParseTuple(args, "l", &pointer)) {
        return NULL;
    }
    bset = (isl_basic_set *) pointer;
    list = PyList_New(0);
    if (list == NULL) {
        return NULL;
    }
    if (isl_basic_set_foreach_constraint(bset, _append_constraint_to_list, list) == -1) {
        PyErr_SetString(PyExc_RuntimeError,
            "an error occurred in isl_basic_set_foreach_constraint");
        Py_DECREF(list);
        return NULL;
    }
    return list;
}

static int _append_basic_set_to_list(isl_basic_set *bset, void *user) {
    return _append_pointer_to_list((void *) bset, user);
}

static PyObject * isl_set_basic_sets(PyObject *self, PyObject *args) {
    long pointer;
    isl_set *set;
    PyObject *list;
    if (!PyArg_ParseTuple(args, "l", &pointer)) {
        return NULL;
    }
    set = (isl_set *) pointer;
    list = PyList_New(0);
    if (list == NULL) {
        return NULL;
    }
    if (isl_set_foreach_basic_set(set, _append_basic_set_to_list, list) == -1) {
        PyErr_SetString(PyExc_RuntimeError,
            "an error occurred in isl_set_foreach_basic_set");
        Py_DECREF(list);
        return NULL;
    }
    return list;
}

static int _append_point_to_list(isl_point *point, void* user) {
    return _append_pointer_to_list((void *) point, user);
}

static PyObject * isl_set_points(PyObject *self, PyObject *args) {
    long pointer;
    isl_set *set;
    PyObject *list;
    if (!PyArg_ParseTuple(args, "l", &pointer)) {
        return NULL;
    }
    set = (isl_set *) pointer;
    list = PyList_New(0);
    if (list == NULL) {
        return NULL;
    }
    if (isl_set_foreach_point(set, _append_point_to_list, list) == -1) {
        PyErr_SetString(PyExc_RuntimeError,
            "an error occurred in isl_set_foreach_point");
        Py_DECREF(list);
        return NULL;
    }
    return list;
}

static int _append_vertex_to_list(isl_vertex *vertex, void* user) {
    return _append_pointer_to_list((void *) vertex, user);
}

static PyObject * isl_vertices_vertices(PyObject *self, PyObject *args) {
    long pointer;
    isl_vertices *vertices;
    PyObject *list;
    if (!PyArg_ParseTuple(args, "l", &pointer)) {
        return NULL;
    }
    vertices = (isl_vertices *) pointer;
    list = PyList_New(0);
    if (list == NULL) {
        return NULL;
    }
    if (isl_vertices_foreach_vertex(vertices, _append_vertex_to_list, list) == -1) {
        PyErr_SetString(PyExc_RuntimeError,
            "an error occurred in isl_vertices_foreach_vertex");
        Py_DECREF(list);
        return NULL;
    }
    return list;
}


static PyMethodDef _islhelper_methods[] = {
    {"isl_basic_set_constraints", isl_basic_set_constraints, METH_VARARGS, NULL},
    {"isl_set_basic_sets", isl_set_basic_sets, METH_VARARGS, NULL},
    {"isl_set_points", isl_set_points, METH_VARARGS, NULL},
    {"isl_vertices_vertices", isl_vertices_vertices, METH_VARARGS, NULL},
    {NULL, NULL, 0, NULL}
};

static struct PyModuleDef _islhelpermodule = {
    PyModuleDef_HEAD_INIT,
    "_islhelper",
    NULL,
    0,
    _islhelper_methods
};

PyMODINIT_FUNC PyInit__islhelper(void) {
    PyObject *m;
    m = PyModule_Create(&_islhelpermodule);
    if (m == NULL) {
        return NULL;
    }
    if (PyModule_AddObject(m, "dim_set", PyLong_FromLong(isl_dim_set)) == -1) {
        return NULL;
    }
    return m;
}
