import graphlab as _graphlab
import graphlab.connect as _mt
from graphlab.toolkits import _supervised_learning as _sl

def create(dataset, target, features=None, verbose=True):
    """
    Automatically create a suitable regression model based on the provided
    training data.

    To use specific options of a desired model, use the ``create`` function
    of the corresponding model.

    Parameters
    ----------
    dataset : SFrame
        A training dataset containing feature columns and a target column.
        Only numerical typed (int, float) target column is allowed.

    target : str
        The name of the column in ``dataset`` that is the prediction target.
        This column must have a numeric type.

    features : list[str], optional
        A list of columns names of features used for training the model.
        Defaults to None (uses all columns in the input SFrame).

    verbose : boolean, optional
        If True, print progress information during training.

    Returns
    -------
      out : A trained regression model.

    See Also
    --------
    graphlab.linear_regression.LinearRegression, 
    graphlab.boosted_trees_regression.BoostedTreesRegression

    Examples
    --------
    .. sourcecode:: python

      # Setup the data
      >>> import graphlab as gl
      >>> data =  gl.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

      # Selects the best model based on your data.
      >>> model = gl.regression.create(data, target='price', 
      ...                                  features=['bath', 'bedroom', 'size'])

      # Make predictions and evaluate results.
      >>> predictions = model.predict(data)
      >>> results = model.evaluate(data)

      # Setup the data
      >>> import graphlab as gl
      >>> data =  gl.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

      # Selects the best model based on your data.
      >>> model = gl.regression.create(data, target='price', 
      ...                                  features=['bath', 'bedroom', 'size'])

      # Make predictions and evaluate results.
      >>> predictions = model.predict(data)
      >>> results = model.evaluate(data)

    """
    _mt._get_metric_tracker().track('toolkit.regression.create')
    return _sl.create_with_model_selector(dataset, target, 
              model_selector = _graphlab.extensions._regression_model_selector,
              features = features, 
              verbose = verbose)
