import graphlab as _graphlab
import graphlab.connect as _mt
from graphlab.toolkits import _supervised_learning as _sl

def create(dataset, target, features=None, verbose=True):
    """
    Automatically create a suitable classifier model based on the provided
    training data.

    To use specific options of a desired model, use the ``create`` function
    of the corresponding model.

    Parameters
    ----------
    dataset : SFrame
        A training dataset containing feature columns and a target column.
        Only numerical typed (int, float) target column is allowed.

    target : str
        The name of the column in ``dataset`` that is the prediction target.
        This column must have a numeric type.

    features : list[str], optional
        A list of columns names of features used for training the model.
        Defaults to None (uses all columns in the input SFrame).

    verbose : boolean, optional
        If True, print progress information during training.

    Returns
    -------
      out : A trained classifier model.

    See Also
    --------
    graphlab.boosted_trees_classifier.BoostedTreesClassifier, 
    graphlab.logistic_classifier.LogisticClassifier, 
    graphlab.svm_classifier.SVMClassifier, 
    graphlab.neuralnet_classifier.NeuralNetClassifier

    Examples
    --------
    .. sourcecode:: python

      # Setup the data
      >>> import graphlab as gl
      >>> data =  gl.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')
      >>> data['is_expensive'] = data['price'] > 30000
      
      # Selects the best model based on your data.
      >>> model = gl.regression.create(data, target='price', 
      ...                                  features=['bath', 'bedroom', 'size'])

      # Make predictions and evaluate results.
      >>> predictions = model.classify(data)
      >>> results = model.evaluate(data)

    """
    _mt._get_metric_tracker().track('toolkit.classifier.create')
    return _sl.create_with_model_selector(dataset, target, 
              model_selector = _graphlab.extensions._classifier_model_selector,
              features = features, 
              verbose = verbose)
