import copy as _copy
import os as _os
import logging as _logging

from graphlab.connect.aws._ec2 import DEFAULT_INSTANCE_TYPE as _DEFAULT_INSTANCE_TYPE, get_credentials as _get_credentials
import graphlab as _gl

class _Enum(set):
    def __getattr__(self, name):
        if name in self:
            return name
        raise AttributeError

_ENVIRONMENT = _Enum(['Local', 'LocalAsynchronous', 'EC2', 'Hadoop'])
__LOGGER__ = _logging.getLogger(__name__)

class Environment(object):
    """Base class for environments for running Tasks / Jobs."""

    def __init__(self, name, env):
        """
        Constructor for base Environment, should not be instantiated directly.
        """
        if name is None:
            raise TypeError("Name is required when creating an Environment.")

        if not isinstance(name, str):
            raise TypeError("Name is required to be a string.")

        if env not in _ENVIRONMENT:
            raise Exception("Invalid type of environment. Environments must be one of %s" % ENVIRONMENT)

        self._session = _gl.deploy._default_session
        self.params = dict()
        self.name = name
        self.env = env
        self._modified_since_last_saved = None
        self._typename = Environment.__name__

        self._session.register(self)

    def get_max_degree_of_parallelism(self):
        """Return the number of tasks that can be executed in parallel."""
        raise NotImplementedError

    def set_params(self, params):
        """
        Set params dictionary for this environment.

        This parameter dictionary is for things that might be different between
        environments (like database hosts, credentials, metrics publishing
        locations, etc).

        Parameters
        ----------
        params : dict
            Basic python dict (string and number types) for additional
            parameters for this environment.
        """
        self.params.update(params)
        self._set_dirty_bit()
        return self

    def get_params(self):
        """Return the params dictionary for this environment."""
        return self.params

    def _set_dirty_bit(self):
        if self._modified_since_last_saved is not None and not self._modified_since_last_saved:
            self._modified_since_last_saved = True

    def clone(self):
        """
        Clones environment and returns the cloned environment.

        Returns
        -------
        out
            Cloned environment.

        Notes
        -----
        Cloned environment cannot be saved unless it is renamed.
        """
        return _copy.deepcopy(self)

    def save(self):
        """Saves environment."""
        self._session.save(self, typename=self._typename)
        __LOGGER__.info("Environment saved successfully.")

    def __getstate__(self):
        odict = dict.copy(self.__dict__)
        if hasattr(odict, '_session'):
            del odict._session
        return odict

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        return "Environment: [name: %s, type: %s, params: %s]" % (self.name, self.env, self.params)

    def __to_dict__(self):
        json_dict = {}
        json_dict["name"] = self.name
        json_dict["type"] = self.env
        json_dict["params"] = self.params
        return json_dict

class LocalAsync(Environment):
    """
    Environment for executing jobs in the background (asynchronously) on this machine.

    Parameters
    ----------
    name : str
        The name for the environment.

    Examples
    --------
    >>> async = graphlab.deploy.environment.LocalAsync('async')
    """
    def __init__(self, name):
        super(LocalAsync, self).__init__(name, _ENVIRONMENT.LocalAsynchronous)

    def __str__(self):
        return "Local (asynchronous): [name: %s, params: %s]" % (self.name, self.params)

    def get_max_degree_of_parallelism(self):
        """Return the number of tasks that can be executed in parallel."""
        return 1

class Local(Environment):
    """
    Environment for executing jobs on this machine.

    Parameters
    ----------
    name : str
        The name for the environment.

    Examples
    --------
    >>> local = graphlab.deploy.environment.Local('local')
    """

    def __init__(self, name):
        super(Local, self).__init__(name, _ENVIRONMENT.Local)

    def __str__(self):
        return "Local: [name: %s, params: %s]" % (self.name, self.params)

    def get_max_degree_of_parallelism(self):
        """Return the number of tasks that can be executed in parallel."""
        return 1

class EC2(Environment):
    """
    Environment for executing jobs on AWS EC2.

    Parameters
    ----------
    name : str
        The name for the environment.

    s3_log_folder_path : str
        The S3 path to folder where the log file will be saved. Format should be: 's3://bucket_name/path/'

    aws_access_key : str, optional
        The AWS Access Key to use to launch the host(s). This parameter must be set or the
        AWS_ACCESS_KEY_ID environment variable must be set.

    aws_secret_key : str, optional
        The AWS Secret Key to use to launch the host(s).  This parameter must be set or the
        AWS_SECRET_ACCESS_KEY environment variable must be set.

    region : str, optional
        The AWS region in which to launch your instance. Default is 'us-west-2'.

    instance_type : str, optional
        The EC2 instance type to launch, default is m3.xlarge. We support all instance types
        except: 't2.micro', 't2.small', 't2.medium' and 'm3.medium'. For a list of instance_types,
        please refer to `here <http://aws.amazon.com/ec2/instance-types/#Instance_Types>`_.

    security_group : str, optional
        The name of the security group for the EC2 instance to use.

    tags : dict, optional
        A dictionary containing the name/value tag pairs to be assigned to the instance. If you want
        to create only a tag name, the value for that tag should be the empty string (i.e. '').
        In addition to these specified tags, a 'GraphLab' tag will also be assigned.

    CIDR_rule : str | list [str], optional
        The Classless Inter-Domain Routing rule(s) to use for the instance. Useful for restricting the IP
        Address Range for a client. Default is no restriction. If you specify CIDR_rule(s), you must also
        specify a security group to use.

    Examples
    --------
    >>> ec2 = graphlab.deploy.environment.EC2('ec2_env_name', 's3://bucket/path')

    """
    def __init__(self, name, s3_log_folder_path, aws_access_key = None, aws_secret_key = None,
                 region = 'us-west-2', instance_type = _DEFAULT_INSTANCE_TYPE,
                 security_group = None, tags = None, CIDR_rule = None):
        super(EC2, self).__init__(name, _ENVIRONMENT.EC2)

        if(aws_access_key is not None and aws_secret_key is not None):
            self.aws_access_key, self.aws_secret_key = aws_access_key, aws_secret_key
        else:
            if bool(aws_access_key) != bool(aws_secret_key):
                raise IOError('You must specify both aws_access_key and aws_secret_key.')
            try:
                _get_credentials()
            except:
                __LOGGER__.warning('No AWS credentials were passed in and none are set globally.' \
                                       ' In order to use this environment, you will need to' \
                                       ' set your credentials globally using' \
                                       ' graphlab.aws.set_credentials(...).'
                                   )

        if not isinstance(region, str):
            raise TypeError('region must be a string')

        if not isinstance(instance_type, str):
            raise TypeError('instance type must be a string')

        if security_group and not isinstance(security_group, str):
            raise TypeError('instance type must be a string')

        if tags and not isinstance(tags, dict):
            raise TypeError('tags must be a dict')

        if CIDR_rule and not (isinstance(CIDR_rule, str) or isinstance(CIDR_rule, list)):
            raise TypeError('CIDR_rule must be a string or a list of strings')

        (self.s3_bucket, self.s3_log_folder_path) = EC2._parse_s3_path(s3_log_folder_path)

        self.region = region
        self.instance_type = instance_type

        self.security_group = security_group
        self.tags = tags if tags else {}
        self.CIDR_rule = CIDR_rule


    @staticmethod
    def _parse_s3_path(s3_log_path):
        if not s3_log_path.startswith('s3://'):
            raise TypeError('s3_log_path must be an S3 path: s3://bucket_name/path')

        tokens = s3_log_path.split('/')
        bucket_name = tokens[2]

        s3_folder = '/'.join(tokens[3:])
        if s3_folder and s3_folder[-1] != '/':
            s3_folder = s3_folder + '/'

        return (bucket_name, s3_folder)


    def __str__(self):
        return "EC2: [name: %s, access_key: %s, instance_type: %s, region: %s, params: %s, log folder: s3://%s/%s]" % \
               (self.name, self.aws_access_key if hasattr(self, 'aws_access_key') else 'None',
                self.instance_type, self.region, self.params, self.s3_bucket, self.s3_log_folder_path)

    def get_max_degree_of_parallelism(self):
        """Return the number of tasks that can be executed in parallel."""
        if hasattr(self, 'num_hosts'):
            return self.num_hosts
        return 1


class Hadoop(Environment):
    """
    Environment for running from a Hadoop YARN cluster.

    Parameters
    ----------
    name : str
        The name of this environment.

    config_dir : str
        The location, on the local filesystem, of the hadoop config directory.

    memory_mb : int
        The memory in MB required for job execution.  Default is 4096.

    gl_source : None | 'none' | HDFS full path | 'native'
        The location of a tar.gz containing GraphLab Create and all its requirements.

        Options:
            None | 'none' : The GraphLab client will download and package
                GraphLab binaries and all required dependencies and place them
                on hdfs.  virtualenv-2.7 is required on the path.

            <hdfs url> : The GraphLab client will use a tar.gz file available
                on hdfs as the source for GraphLab binaries and all
                dependencies.

            'native' : The GraphLab client will use a native GraphLab Create
                installation on the hadoop cluster.


    Notes
    -----
    - Requires 'hadoop' and 'yarn' commands to be available on the local path.

    Examples
    --------
    >>> hd = graphlab.deploy.environment.Hadoop('hd')

    """

    def __init__(self, name, config_dir=None, memory_mb=4096, virtual_cores=2, gl_source=None):
        super(Hadoop, self).__init__(name, _ENVIRONMENT.Hadoop)
        if not (gl_source in ['none', None, 'native', 'Native'] or gl_source.startswith('hdfs://')):
            raise TypeError("gl_source must be 'Native', None, or HDFS URL.")

        if not gl_source or gl_source=='none':
            self.path = 'none'
        else:
            self.path = gl_source

        if not virtual_cores >=2:
            raise TypeError("Virtual cores must be greater than or equal to two.")

        self.config_dir = config_dir
        self.container_memory = memory_mb
        self.vcores = virtual_cores

    def set_graphlab_source(self, path):
        """
        Set the path for GraphLab Create and its dependency archive.

        This should not be needed to be set, as the default behavior is to
        download GraphLab Create from the machine submitting a Job to Hadoop Environment.

        Parameters
        ----------
        path : 'none' | HDFS path | local OS path | 'native'
            Path as a string to a .tar.gz file that captures GraphLab Create
            Python egg and all dependency eggs as well. To build this file a
            simple way is to use pip --download graphlab-create and then take
            the downloaded .tar.gz files and archive them into one .tar.gz.

            Options:
                None | 'none' : The GraphLab client will download and package
                    GraphLab binaries and all required dependencies and place them
                    on hdfs.  virtualenv-2.7 is required on the path.

                <hdfs url> : The GraphLab client will use a tar.gz file available
                    on hdfs as the source for GraphLab binaries and all
                    dependencies.

                'native' : The GraphLab client will use a native GraphLab Create
                    installation on the hadoop cluster.

        """
        # TODO valid values are "none", an hdfs path, a local os path, or native
        # if not _os.path.exists(path) or not _os.path.isfile(path):
        #     raise Exception("Invalid Path, must be an existing file. "
        #                     "Should point to GraphLab Create egg (.tar.gz file)")
        if not (path in ['none', None, 'native', 'Native'] or path.startswith('hdfs://')):
            raise TypeError("path must be 'Native', None, or HDFS URL.")
        self.path = path
        self._set_dirty_bit()

    def get_container_memory(self):
        """Return the memory (in MBs) this environment will request from Hadoop."""
        return self.container_memory

    def get_max_degree_of_parallelism(self):
        """Return the number of tasks that can be executed in parallel."""
        return self.vcores

    def set_container_memory(self, memory_mb):
        """Set the memory (in MBs) this environment will request from Hadoop."""
        self.container_memory = memory_mb
        self._set_dirty_bit()

    def get_config_dir(self):
        """Return the YARN config directory."""
        return self.config_dir

    def set_config_dir(self, config_dir):
        """Set the YARN config directory."""
        self.config_dir = config_dir
        self._set_dirty_bit()

    def get_graphlab_source(self):
        """Return the path to the gl_source archive (.tar.gz)."""
        return str(self.path)

    def set_virtual_cores(self, num_cores):
        """Set the number of cores to request from Hadoop."""
        self.vcores = num_cores
        self._set_dirty_bit()

    def get_virtual_cores(self):
        """Return the number of cores this environment will request from Hadoop."""
        return str(self.vcores)

    def __str__(self):
        return "Hadoop: [name: %s, config_dir: %s, container_memory: %s," \
                "virtual_cores: %s, gl_source: %s, params: %s]" % (self.name,
                self.config_dir, self.container_memory, self.vcores, self.get_graphlab_source(), self.params)

