"""

This package defines the GraphLab PredictiveObject, which is an abstract class
that defines the interfaces for a PredictiveObject. It also provides common
methods used by PredictiveObject, like load, save, validate etc.

"""
import abc
import graphlab
import graphlab_util.cloudpickle as _cloudpickle
import cPickle as _pickle
import os
import json
import tempfile
import tarfile
import shutil
import logging
import _file_util as fu

__logger__ = logging.getLogger(__name__)

class PredictiveObject(object):
  '''Predictive Object Abstraction

  Derived class needs to implement the following interface:
    query(input)
  where input would be a dictionary that is interpreted by the derived class,
  and query returns a dictionary that would be interpreted by client

  '''

  __metaclass__ = abc.ABCMeta

  def __init__(self, description = ''):
    self._dependencies = {}
    self.description = description

  def __repr__(self):
    """Returns a string description of the object."""
    ret = ""
    ret = ret + 'class:' + self._get_full_class_name() + "\n"
    ret = ret + 'description:' + self.description + "\n"
    ret = ret + 'dependencies:' + "\n"
    for (key, value) in self.dependencies.iteritems():
      ret  = ret + "\t" + key + ":" + value.__str__() + "\n"
    return ret

  def __str__(self):
    """Returns a string description of the object."""
    return self.__repr__()

  @property
  def description(self):
    '''The description for the predictive object'''
    return self._description

  @description.setter
  def description(self, description):
    '''Set the description of the predictive object'''
    self._description = description

  @property
  def dependencies(self):
    """Returns all dependencies of this predictive object.

    Returns a dictionary with key begin the unique identifier of the dependent
    object and value being the reference to a GraphLab object (Model, SFrame,
    SGraph, SArray, etc.)
    """
    return self._dependencies

  @dependencies.setter
  def dependencies(self, dependencies):
    '''Set the dependencies of the predictive object'''
    if type(dependencies) != dict:
      raise RuntimeError("Predictive Object dependencies has to be a dict object")
    self._dependencies = dependencies

  def add_dependency(self, uri, dependency):
    '''Adds a dependency to a predictive object.'''
    self._dependencies.update({uri: dependency})

  @staticmethod
  def _validate_query_input(input):
    if type(input) != dict:
      raise TypeError("Input to ModelPredictiveObject has to be a dictionary")

    if not input.has_key('method'):
      raise ValueError('Cannot find key "method" in input to ModelPredictiveObject')

    if not input.has_key('data'):
      raise ValueError('Cannot find key "data" in input to ModelPredictiveObject')

  @abc.abstractmethod
  def query(self, input):
    """Given input, get the prediction for the input"""
    pass

  def get_doc_string(self):
    '''Returns documentation for the predictive object query

    By default, this method returns the docstring for 'query' method
    Derived class may overwrite this method to dynamically create doc string
    '''
    if self.query.__doc__ != None:
      return self.query.__doc__
    else:
      return "-"

  def post_load(self):
    """After a predictive object is loaded, do object specific post-load process
    Implemented by derived class
    """
    pass

  def save(self, path, aws_credentials = {}):
    """ Save predictive object to the given path

    Parameters
    ----------
    path : str
      The location to save the predictive object to
    """
    __logger__.info('Saving predictive object to %s ' % (path))

    # only support saving to local or S3 for now
    if (not fu.is_s3_path(path)) and (not fu.is_local_path(path)):
      raise RuntimeError("Only save to local and S3 path is supported, cannot \
        save predictive object to path %s. " % path)

    if (fu.is_s3_path(path)):
      self._save_s3(path, aws_credentials)
    else:
      self._save_local(path)

  def _save_s3(self, path, aws_credentials):
    '''Save current predictive object to S3

    Here are steps involved in saving a predictive object
      step1: create a temporary directory to save the files
      step2: zip the files to a temporary tar.gz file
      step3: upload to s3
    '''
    fd, temp_zip_file = tempfile.mkstemp()
    try:
      self._save_local(temp_zip_file)
      fu.upload_to_s3(temp_zip_file, path, aws_credentials)
    finally:
      os.remove(temp_zip_file)

  def _save_local(self, path):
    '''Save current predictive object to local path

    Here are steps involved in saving a predictive object
      step1: create a temporary directory to save the files
      step2: zip the files to target path designated by path
    '''
    if os.path.exists(path):
      if os.path.isfile(path):
        __logger__.warning("Overwriting existing file '%s' when saving predictive object" % path)
      else:
        raise RuntimeError("Path %s already exists, please remove that and save again" % path)

    temp_directory = tempfile.mkdtemp()

    # put all files under a known directory so that we know how to access the
    # content later
    po_obj_path = os.path.join(temp_directory, 'definition')

    try:
      # save to a temp directory first
      self._save_imp(po_obj_path)

      # zip the directory to target file
      fu.zip_directory(po_obj_path, path)
    finally:
      shutil.rmtree(temp_directory)

  def __getstate__(self):
    odict = self.__dict__.copy()
    del odict['_description']
    del odict['_dependencies']
    return odict

  def _save_imp(self, po_path):
    '''Save the predictive object to a directory

      The files for a predictive object is laid out the following way:

        po_path/describe.json -- serialized information about the predictive
          object, including description, dependencies, etc,
          in json string format
        po_path/definition -- cloudpickle-serialized PredictiveObject
        po_path/GraphLabObjects -- all dependent GraphLab objects, each in its
          own directory:
          po_path/GraphLabObjects/uri1/ -- serialized GraphLab object with uri1
          po_path/GraphLabObjects/uri2/ -- serialized GraphLab object with uri2
      '''
    fu.create_directory(po_path)

    describe = {
      'description': self.description,
      'dependencies':{}
    }

    # save all dependent objects
    graphlab_obj_path = self._get_graphlab_object_path(po_path)
    fu.create_directory(graphlab_obj_path)

    for (uri, gl_obj) in self.dependencies.iteritems():

      # persiste the graphlab object
      gl_object_path = os.path.join(graphlab_obj_path, uri)
      fu.create_directory(gl_object_path)

      gl_obj.save(gl_object_path)

      # add to the global describe dictionary
      describe['dependencies'][uri] = {
        'path': os.path.relpath(po_path, gl_object_path),
        'type': self._get_graphlab_object_type(gl_obj),
      }

    # persist the global description
    describe_path = self._get_describe_path(po_path)
    self._save_object(describe_path, describe)

    # persist the definition of myself
    definition_path = self._get_definition_path(po_path)
    try:
      with open(definition_path, 'wb') as f:
        _cloudpickle.dump(self, f)
    except Exception as e:
      __logger__.error('Unable to save object: %s' % (e.message))

  @classmethod
  def load(cls, path):
    """ Load predictive object from given path
    """
    new_po = None
    if (fu.is_s3_path(path)):
      new_po = cls._load_s3(path)
    else:
      new_po = cls._load_local(path)

    # call derived class post_load
    new_po.post_load()
    return new_po

  @classmethod
  def _load_local(cls, path):
    path = fu.expand_full_path(path)

    if not os.path.isfile(path):
      raise RuntimeError("path %s is not a file, a tar.gz file is expected." % path)

    po_path = tempfile.mkdtemp()
    fu.unzip_directory(path, po_path)

    po_object_path = os.path.join(po_path, 'definition')

    # load the describe
    describe = cls._load_object(cls._get_describe_path(po_object_path))

    po_obj = None

    # load the instance
    try:
      with open(cls._get_definition_path(po_object_path), 'rb') as f:
        po_obj = _pickle.load(f)
    except Exception as e:
      __logger__.error('Unable to load object: %s' % (e.message))
      raise

    po_obj.description = str(describe['description'])

    __logger__.info('Loaded predictive object of type: %s' % (type(po_obj)))

    # load all dependencies
    graphlab_obj_path = cls._get_graphlab_object_path(po_object_path)
    dependencies = describe['dependencies']
    gl_objects = {}
    for (uri, gl_obj_info) in dependencies.iteritems():
      uri = str(uri)
      gl_obj_type = gl_obj_info['type']
      gl_obj_path = os.path.join(graphlab_obj_path, uri)
      gl_objects[uri] = cls._load_graphlab_object(gl_obj_type, gl_obj_path)

    po_obj.dependencies = gl_objects

    return po_obj

  @classmethod
  def _load_s3(cls, path):
    fd, temp_zip_file = tempfile.mkstemp()
    try:
      fu.download_from_s3(path, temp_zip_file)
      return cls._load_local(temp_zip_file)
    finally:
      os.remove(temp_zip_file)

  @classmethod
  def _get_graphlab_object_path(cls, root):
    return os.path.join(root, 'GraphLabObjects')

  @classmethod
  def _get_describe_path(cls, root):
    return os.path.join(root, 'meta')

  @classmethod
  def _get_definition_path(cls, root):
    return os.path.join(root, 'definition')

  @classmethod
  def _is_graphlab_object(cls, obj):
    return isinstance(obj, graphlab.Model) | \
      isinstance(obj, graphlab.SFrame) | \
      isinstance(obj, graphlab.SArray) | \
      isinstance(obj, graphlab.SGraph)

  @classmethod
  def _get_graphlab_object_type(cls, obj):
    if isinstance(obj, graphlab.Model):
      return 'model'
    elif isinstance(obj, graphlab.SArray):
      return 'sarray'
    elif isinstance(obj, graphlab.SGraph):
      return 'sgraph'
    elif isinstance(obj, graphlab.SFrame):
      return 'sframe'
    else:
      raise TypeError("Dependent object is not a GraphLab object")

  @classmethod
  def _load_graphlab_object(cls, obj_type, obj_path):
    if obj_type == 'model':
      return graphlab.load_model(obj_path)
    elif obj_type == 'sarray':
      return graphlab.SArray(obj_path)
    elif obj_type == 'sframe':
      return graphlab.load_sframe(obj_path)
    elif obj_type == 'sgraph':
      return graphlab.load_sgraph(obj_path)
    else:
      raise RuntimeError(str(obj_type) + ' is not supported')

  @classmethod
  def _save_object(cls, filename, content):
    '''Serialize a Python object using json serializer and save to a file'''
    serialized  = json.dumps(content)
    with open(filename, 'w') as f:
      f.write(serialized)

  @classmethod
  def _load_object(cls, filename):
    '''Load a serialized json object from file
    Returns the Python object deserialized from the json string
    '''
    with open(filename, 'r') as f:
      s = f.read()
      return json.loads(s)

  def _get_full_class_name(self):
    module = self.__class__.__module__
    if module is None or module == str.__class__.__module__:
        return self.__class__.__name__
    return module + '.' + self.__class__.__name__

  @classmethod
  def _make_serializable(cls, result):
    '''Convert a result from model query to python data structure that can
    easily serialize over network
    '''
    ret = None
    if type(result) == graphlab.SFrame:
      # SFrame is converted to list of dictionary
      if len(result.column_names()) > 1:
        ret = list(result.pack_columns(dtype=dict)['X1'])
      else:
        col = result.column_names()[0]
        ret = [{col: i} for i in result[col]]
    elif type(result) == graphlab.SArray:
      # SArray is converted to list directly
      ret = list(result)
    elif type(result) == graphlab.SGraph:
      raise RuntimeError("SGraph is not supported as query result")
    else:
      # otherwise original value
      ret = result

    try:
      s = json.dumps(ret)
    except:
      raise TypeError("Result from model query is not json serializable: %s" % ret)

    return ret
