'''

This package provides commonly used methods for dealing with file operation,
including working with network file system like S3, http, etc.

'''
import os
import time
import boto
import boto.s3
import tarfile
import logging
import shutil

__logger__ = logging.getLogger(__name__)

__RETRY_TIMES = 5
__SLEEP_SECONDS_BETWEEN_RETRIES = 2

def get_protocol(path):
  '''Given a path, returns the protocol the path uses

  For example,
    's3://a/b/c/'  returns 's3'
    'http://a/b/c' returns 'http'
    'tmp/a/bc/'    returns ''

  '''
  pos = path.find('://')
  if pos < 0:
    return ''
  return path[0:pos].lower()

def is_local_path(path):
  '''Returns True if the path indicates a local path, otherwise False'''
  protocol = get_protocol(path)
  return protocol != 'hdfs' and protocol != 's3' and \
        protocol != 'http' and protocol != 'https'

def is_s3_path(path):
  '''Returns True if the path indicates a s3 path, otherwise False'''
  protocol = get_protocol(path)
  return protocol == 's3'

def create_directory(path, drop_if_exist=True):
  '''Create a clean directory for use

  If the directory already exists, remove existing directory and create a new one.
  If the directory does not exists, create a new directory
  '''
  if not is_local_path(path):
    raise RuntimeError("Path %s has to be a local path" % path)

  path = expand_full_path(path)

  if os.path.exists(path):
    if (drop_if_exist):
      __logger__.warning("Overwriting existing directory: %s" % path)
      shutil.rmtree(path)
    else:
      raise RuntimeError("Directory %s already exists" % path)

  __logger__.debug("creating directory: %s" % path)
  os.makedirs(path)

def upload_to_s3(local_file, s3_path, aws_credentials):
  '''Upload a local file to s3
  '''
  __logger__.info('Uploading local file %s to s3 path: %s' % (local_file, s3_path))

  if not os.path.exists(local_file):
    raise RuntimeError("Cannot find file: %s" % local_file)

  if not os.path.isfile(local_file):
    raise RuntimeError("Path %s is not a file" % local_file)

  if not is_s3_path(s3_path):
    raise RuntimeError("Path %s is not a valid s3 path" % s3_path)

  k = _get_s3_key(s3_path, aws_credentials)

  # now upload
  num_retries = 0
  while num_retries < __RETRY_TIMES:
    try:
      k.set_contents_from_filename(
        local_file,
        cb = _s3_callback,
        num_cb=10)

      __logger__.info("Successfully uploaded file to s3 path %s" % s3_path)
      break
    except Exception as e:
      num_retries = num_retries + 1
      __logger__.info("Error hit while uploading to S3: %s" % e)
      __logger__.info("Retrying %s out of %s" % (num_retries, __RETRY_TIMES))
      if num_retries == __RETRY_TIMES:
        raise e
      time.sleep(__SLEEP_SECONDS_BETWEEN_RETRIES)

def download_from_s3(s3_path, local_file):
  '''Download a file from S3'''

  local_file = expand_full_path(local_file)

  __logger__.info('Downloading file %s from s3 to local file %s' % (s3_path, local_file))

  if not is_local_path(local_file):
    raise RuntimeError("local_file path is not correct: %s" % local_file)

  if os.path.exists(local_file):
    __logger__.warning("Overwriting file %s" % local_file)

  if not is_s3_path(s3_path):
    raise RuntimeError("Path %s is not a valid S3 path" % s3_path)


  # now download
  k = _get_s3_key(s3_path)
  num_retries = 0
  while num_retries < __RETRY_TIMES:
    try:
      if not k.exists():
        raise RuntimeError("Key %s does not exist." % s3_path)

      k.get_contents_to_filename(
        local_file,
        cb = _s3_callback,
        num_cb = 10)

      # check the size of the file, if 0, try again
      if os.path.getsize(local_file) == 0:
        raise RuntimeError("Download file size is 0, try again.")

      __logger__.info("Successfully downloaded file %s from s3" % s3_path)
      break
    except Exception as e:
      num_retries = num_retries + 1
      __logger__.info("Error hit while download from S3: %s" % e)
      __logger__.info("Retrying %s out of %s" % (num_retries, __RETRY_TIMES))
      if num_retries == __RETRY_TIMES:
        raise e
      time.sleep(__SLEEP_SECONDS_BETWEEN_RETRIES)


def zip_directory(directory_name, out_file_path):
  '''Pack a directory to one file tar.gz file

  Parameters
  ----------
  directory_name : str
    The directory to be packed

  out_file_path : str
    The output file name
  '''

  directory_name = expand_full_path(directory_name)
  out_file_path = expand_full_path(out_file_path)

  if not os.path.exists(directory_name):
    raise RuntimeError("directory %s does not exist." % directory_name)

  if not os.path.isdir(directory_name):
    raise RuntimeError("Path %s is not a directory." % directory_name)

  if not is_local_path(out_file_path):
    raise RuntimeError("out_file_path must be a local path")

  if os.path.exists(out_file_path):
    __logger__.warning("Overwriting file %s" % out_file_path)

  with tarfile.open(out_file_path, "w:gz") as tar:
    tar.add(directory_name, arcname=os.path.basename(directory_name))

def unzip_directory(zip_file_path, directory_name):
  '''Unpack a tar.gz file into a directory
  '''
  zip_file_path = expand_full_path(zip_file_path)
  directory_name = expand_full_path(directory_name)

  if not os.path.isfile(zip_file_path):
    raise RuntimeError("File %s does not exists." % zip_file_path)

  with tarfile.open(zip_file_path) as tar:
    tar.extractall(path=directory_name)

def parse_s3_path(path):
  '''Parse a s3 path to bucket name and path name

  Parameters
  -----------
  path :  str
    s3 path like: s3://bucket_name/path/to/somewhere

  Returns
  --------
  out : (bucket_name, path_name)
  '''
  if not is_s3_path(path):
      raise ValueError('path is not a s3 path: %s' % path)

  tokens = path.split('/')
  bucket_name = tokens[2]

  s3_directory = '/'.join(tokens[3:])

  # remove trailing '/' if exists
  if s3_directory and s3_directory[-1] == '/':
      s3_directory = s3_directory[0:-1]

  return (bucket_name, s3_directory)

def s3_delete_key(s3_bucket_name, s3_key_name, aws_credentials = {}):
  conn = boto.connect_s3(**aws_credentials)
  bucket = conn.get_bucket(s3_bucket_name)
  bucket.delete_key(s3_key_name)

def s3_recursive_delete(s3_path, aws_credentials = {}):
  (s3_bucket_name, s3_key_prefix) = parse_s3_path(s3_path)
  conn = boto.connect_s3(**aws_credentials)
  bucket = conn.get_bucket(s3_bucket_name)
  matches = bucket.list(prefix=s3_key_prefix)
  bucket.delete_keys([key.name for key in matches])

def expand_full_path(path):
  '''Expand a relative path to a full path

  For example,
    '~/tmp' may be expanded to '/Users/username/tmp'
    'abc/def' may be expanded to '/pwd/abc/def'
  '''
  return os.path.abspath(os.path.expanduser(path))

def _get_s3_key(s3_path, aws_credentials = {}):
  '''Given S3 path, get the key object that represents the path'''
  conn = boto.connect_s3(**aws_credentials)
  (bucket_name, path) = parse_s3_path(s3_path)
  bucket = conn.get_bucket(bucket_name)
  k = boto.s3.key.Key(bucket)
  k.key = path
  return k

def _s3_callback(complete, total):
  if complete > 0:
    __logger__.info('%d%% completed.' % int(1.0 * complete/total * 100))
