import graphlab.canvas
import graphlab.canvas.views.base
import graphlab.deploy
import json

class PredictiveService(graphlab.canvas.views.base.BaseView):
    def __init__(self, obj, params=None):
        super(PredictiveService, self).__init__(obj, params)
        self.objectType = 'PredictiveService'
        self.register_handler('post', 'query', self.__query_handler)
        self.register_handler('get', 'get_ps_status', self.__get_ps_status)
        self.__cache_ps = None

    def get_metadata(self):
        ps_info = self.__get_ps_info()
        data = {
            'ps_endpoint' : self.params['endpoint'],
            'ps_name' : self.obj,
            'ps_info' : ps_info,
            'pslistName' : 'Predictive Service'
        }
        return data

    def __get_ps_info(self):
        name = self.obj 
        ps = self.__load_ps_object()
        # return to session page if loaded ps none
        if ps is None:
            return {'deleted': True} 

        ps_info = {}
        
        # get deployed and pending predictive objects
        ps_info['deployed'] = ps.deployed_predictive_objects
        ps_info['pending'] = ps.pending_changes

        # get metadata on the predictive services
        ps_info['s3_endpoint'] = self.params['endpoint']
        ps_info['name'] = name
        ps_info['description'] = ps._description if ps._description else ""
        ps_info['api_key'] = ps._api_key if ps._api_key else ""

        # get query endpoint (LB dns name)
        try:
            ps_info['lb_dns_name'] = ps._environment.load_balancer_dns_name
        except Exception as e:
            ps_info['lb_dns_name'] = "N/A"

        return ps_info

    def __get_ps_status(self, url, handler):
        ps = self.__load_ps_object()
        
        # get status list of all instances
        status_info = {}
        try:
            status_list = ps._get_status()
            for status in status_list:
                instance_id = str(status['id'])
                description = str(status['reason'])
                dns = str(status['dns_name'])
                state = str(status['state'])
                status_info[instance_id] = {'description' : description, 'state' : state, 'dns': dns}
        except Exception as e:
            status_info = {}

        # get metrics url
        metrics_url = ''
        try:
            raw_text = ps.get_metrics_url()
            parsed = json.loads(str(raw_text))
            if isinstance(parsed, dict):
                if 'url' in parsed:
                    metrics_url = parsed['url']
        except Exception as e:
            metrics_url = ''

        handler.write({'ps_status': status_info, 'ps_metrics_url': metrics_url})

    def __load_ps_object(self):
        """
        Loads the predictive object by name, from the current session.
        """
        return graphlab.deploy.predictive_services[self.obj]

    def __query_handler(self, url, handler):
        # obtain the json query parameters from canvas
        model = handler.get_argument('model').encode('utf-8')
        data = handler.get_argument('data')
        data_dict = json.loads(data)

        # loads and query the ps with the query parameters
        ps = self.__load_ps_object()
        try:
            resp = ps.query(model, data_dict)
        except BaseException as e:
            resp = "Error when querying the Predictive Service: \n%s" % e.message

        # write back response
        handler.write({"results": resp})

    def child_views(self):
        return {}

    def get_js_file(self):
        return 'predservice'

    def get_temporary_name(self):
        return (str(obj),)
