import graphlab.canvas
import graphlab.canvas.views as views
import graphlab.canvas.views.base
import graphlab.canvas.views.task
import graphlab.canvas.views.sframe
from graphlab.deploy._artifact import Pipeline as _Pipeline

import os as _os
import __builtin__

class JobView(graphlab.canvas.views.base.BaseView):

    def __init__(self, obj, params=None):
        super(JobView, self).__init__(obj, params)
        self.register_handler('get', 'logs', self.__get_logs)
        self.__child_views = {}
        self.__cache_logs = None

    def get_metadata(self):
        tasks = []
        # get list of tasks from Tasks or Pipeline
        if self.obj.tasks is not None and len(self.obj.tasks) > 0:
            task = self.obj.tasks[0]
            if isinstance(task, list):
                tasks = list([stage[0].__to_dict__() for stage in self.obj.tasks]) 
            elif isinstance(task, _Pipeline):
                for step in task.steps:
                    tasks.extend(list([task_step.__to_dict__() for task_step in step]))
                        
        # get results if possible
        has_results = 'results' in self.__child_views
        results = None
        if hasattr(self.obj, "get_results") and not has_results:
            results = self.obj.get_results()
            if results is not None:
                has_results = True
                self.__child_views['results'] = graphlab.canvas.views.sframe.SFrameView(results)
            
        return {
            'jobname' : [self.obj.name],
            'tasks' : tasks,
            'env' : self.obj.environment.__to_dict__(),
            'status' : self.obj.get_status(),
            'metrics' : self.obj.get_metrics(),
            'taskstatus' : self.obj._get_task_status(),
            'jobtype' : __builtin__.type(self.obj).__name__,
            'starttime' : self.obj._get_start_time(),
            'endtime' : self.obj._get_end_time(),
            'has_results' : has_results
        }

    def get_staticdata(self):
        data = {
            'status' : self.obj.get_status(),
        }
        data.update(get_metadata())
        return data

    # return logs for the current job
    def __get_logs(self, url, handler):
        log_file_path = self.obj._get_log_file_path()
        log_data = {'url': '', 'data': ''}
        if log_file_path is None:
            return log_data 
        
        if isinstance(self.obj.environment, graphlab.deploy.environment.Local) or \
           isinstance(self.obj.environment, graphlab.deploy.environment.LocalAsync):
            if self.__cache_logs is None:
                if not _os.path.isfile(log_file_path) and self.__cache_logs is None:
                    log_data['data'] = "Log file path does not exist"
                    return log_data
                
                log_data['url'] = "file://" + log_file_path
                # load max of 1mb log file to python
                if _os.path.getsize(log_file_path) > 1048576L:
                    log_data['data'] = "Log file too large to be displayed. Please use the link to download the log file directly"
                    return log_data
                
                data = self.__load_local_log_file(log_file_path)
                log_data['data'] = data
                status = self.obj.get_status()
                if status == "Completed" or status == "Failed":
                    self.__cache_logs = data
            else:
                log_data['url'] = "file://" + log_file_path
                log_data['data'] = self.__cache_logs
        elif isinstance(self.obj.environment, graphlab.deploy.environment.EC2):
            if self.__cache_logs is None:
                if not isinstance(log_file_path, tuple):
                    log_data['data'] = log_file_path
                    return log_data
            
                log_data['url'] = log_file_path[0]
                # load max of 1mb log file to python
                if log_file_path[1] > 1048576L:
                    log_data['data'] = "Log file too large to be displayed. Please use the link to download the log file directly"
                    return log_data
                    
                data = self.__load_s3_log_file(log_file_path[0])
                log_data['data'] = data
                status = self.obj.get_status()
                if status == "Completed" or status == "Failed":
                    self.__cache_logs = data
            else:
                log_data['url'] = log_file_path[0]
                log_data['data'] = self.__cache_logs

        handler.write({'logfile': log_data})         

    def __load_local_log_file(self, log_file_path):
        data = ""
        try:
            with open(log_file_path, 'r') as f:
                data = f.read() 
            f.close()
        except IOError:
            data = "Unable to read file from %s " % log_file_path
        return data

    def __load_s3_log_file(self, log_file_path_s3):
        import urllib2
        data = ""
        try:
            f = urllib2.urlopen(log_file_path_s3)
            data = f.read()
            f.close()
        except URLError:
            data = "Unable to load s3 log file from %s " % log_file_path_s3
        return data

    def child_views(self):
        return self.__child_views

    def get_js_file(self):
        return 'job'

