"""
This module provides GraphLab Canvas server functionality on top of Tornado.
The InteractiveView uses this server to send data-structure-specific
visualization information to the browser.
"""

import graphlab.canvas.handlers
import graphlab.canvas.state

import datetime
import time
import threading
import tornado.httpserver
import tornado.ioloop
import tornado.netutil
import tornado.web
import uuid

class Server:
    #
    # private variables
    #

    # instance variables
    __thread = None # bg thread that runs Tornado
    __port = None # port Tornado is running on (determined automatically at thread launch)
    __port_lock = threading.RLock()
    __last_ping = None # last ping time from the browser (data check)
    __last_ping_lock = threading.RLock()
    __loop = None

    #
    # public API
    #
    
    # instance variables
    state = None

    def __init__(self, state):
        self.state = state

    # methods
    def ping(self):
        """
        Updates the timestamp for the alive() calculation.
        """
        with self.__last_ping_lock:
            self.__last_ping = datetime.datetime.now()

    def alive(self):
        """
        Returns True if the browser has communicated with the server recently, false otherwise.
        """
        with self.__last_ping_lock:
            if self.__last_ping is None:
                return False
            # if the browser hasn't made a request for data in the last 3 seconds, consider it dead
            return (datetime.datetime.now() - self.__last_ping) < datetime.timedelta(0, 3)

    def get_port(self):
        with self.__port_lock:
            return self.__port

    def start(self):
        """
        Starts the canvas server if it is not already active
        """

        if self.__thread is None:
            self.__thread = threading.Thread(target=self.__run, name='canvas_server')
            # TODO -- if we want to keep the Python process running until the
            # visualization is closed (a la Matplotlib) we can take this out of daemon
            # mode and do our own thread management
            self.__thread.daemon = True
            self.__thread.start()
        # block the thread and wait for the Tornado bg thread to initialize until the port is set
        port = None
        while port is None:
            port = self.get_port()
            if port is None:
                time.sleep(0.05)

    #
    # private API
    #

    def __tornado_log(self, handler):
        pass # do nothing for now, so we can suppress tornado info messages

    def __run(self):
        # use our own IOLoop instead of singleton to avoid clashing with IPython Notebook
        self.__loop = tornado.ioloop.IOLoop()
        application = tornado.web.Application(
            graphlab.canvas.handlers.get_handlers(self, self.state),
            log_function=self.__tornado_log,
            io_loop=self.__loop,
            cookie_secret=uuid.uuid4(),
            xsrf_cookies=True
        )
        # bind to a random available TCP port
        sockets = tornado.netutil.bind_sockets(0, 'localhost')
        server = tornado.httpserver.HTTPServer(application, io_loop=self.__loop)
        server.add_sockets(sockets)
        with self.__port_lock:
            self.__port = sockets[0].getsockname()[:2][1]
        self.__loop.start()
