package Pod::Weaver::Plugin::Ditaa;
$Pod::Weaver::Plugin::Ditaa::VERSION = '0.001003';
# ABSTRACT: include ditaa diagrams in your pod

use Moose;
with 'Pod::Weaver::Role::Dialect';

sub translate_dialect {
   Pod::Elemental::Transformer::Ditaa->new->transform_node($_[1])
}

package Pod::Elemental::Transformer::Ditaa {
$Pod::Elemental::Transformer::Ditaa::VERSION = '0.001003';
use Moose;
   with 'Pod::Elemental::Transformer';

   use Capture::Tiny 'capture';
   use autodie;
   use File::Temp;
   use IPC::System::Simple 'system';
   use IO::All;
   use MIME::Base64;
   use namespace::clean;

   sub transform_node {
      my ($self, $node) = @_;
      my $children = $node->children;

    my $x = 0;

    for (my $i = 0 ; $i < @$children; $i++) {
         my $para = $children->[$i];
         next
           unless $para->isa('Pod::Elemental::Element::Pod5::Region')
           and !$para->is_pod
           and $para->format_name eq 'ditaa';

         my $length = @{$para->children};
         confess 'ditaa transformer expects exec region to contain 1 Data para'
           unless $length == 1
           and $para->children->[0]->isa('Pod::Elemental::Element::Pod5::Data');

         $x++;
         my $text = $para->children->[0]->content;

         my $new_doc = $self->_render_figure(
            figure => $x,
            text => $text,
            b64 => $self->_text_to_b64image($text),
         );

         splice @$children, $i, 1, @{$new_doc->children};
      }

      return $node;
   }

   sub _text_to_b64image {
      my ($self, $text) = @_;

      my $tmp_text = tmpnam();
      my $tmp_img  = tmpnam() . '.png';
      open my $fh, '>', $tmp_text;
      print {$fh} $text;
      close $fh;

      my @cmd = (qw( ditaa -o ), $tmp_text, $tmp_img);
      print STDERR join q( ), @cmd
         if $ENV{DITAA_TRACE};

      my $merged_out = capture { system @cmd };
      print STDERR $merged_out if $ENV{DITAA_TRACE};
      my $image = encode_base64(io->file($tmp_img)->binary->all, '');
      unlink $tmp_text unless $ENV{DITAA_TRACE} && $ENV{DITAA_TRACE} =~ m/keep/;
      unlink $tmp_img unless $ENV{DITAA_TRACE} && $ENV{DITAA_TRACE} =~ m/keep/;

      return $image
   }

   sub _render_figure {
      my ($self, %args) = @_;

      my $new_doc = Pod::Elemental->read_string(
         "\n\n=begin text\n\nFigure $args{figure}\n\n" .
         "$args{text}\n\n=end text\n\n" .
          qq(\n\n=begin html\n\n) .
             qq(<p><i>Figure $args{figure}</i>) .
             qq(<img src="data:image/png;base64,$args{b64}"></img></p>\n\n) .
          qq(=end html\n\n)
      );
      Pod::Elemental::Transformer::Pod5->transform_node($new_doc);
      shift @{$new_doc->children}
        while $new_doc->children->[0]
        ->isa('Pod::Elemental::Element::Pod5::Nonpod');

      return $new_doc
   }

}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Pod::Weaver::Plugin::Ditaa - include ditaa diagrams in your pod

=head1 VERSION

version 0.001003

=head1 SYNOPSIS

In your F<weaver.ini>:

 [@Default]
 [-Ditaa]

In the pod of one of your modules:

 =head1 HOW IT WORKS

 =begin ditaa

    +--------+   +-------+    +-------+
    |        | --+ ditaa +--> |       |
    |  Text  |   +-------+    |diagram|
    |Document|   |!magic!|    |       |
    |     {d}|   |       |    |       |
    +---+----+   +-------+    +-------+
        :                         ^
        |       Lots of work      |
        +-------------------------+

 =end ditaa

=head1 DESCRIPTION

It has often been said that a picture is worth a thousand words.  I find that
sometimes a diagram truly can illuminate your design.  This L<Pod::Weaver>
plugin allows you to put L<ditaa|http://ditaa.sourceforge.net/> diagrams in your
pod and render the image for an html view.  In text mode it merely uses the text
diagram directly.

=head1 IN ACTION

=begin text

Figure 1

   +--------+   +-------+    +-------+
   |        | --+ ditaa +--> |       |
   |  Text  |   +-------+    |diagram|
   |Document|   |!magic!|    |       |
   |     {d}|   |       |    |       |
   +---+----+   +-------+    +-------+
       :                         ^
       |       Lots of work      |
       +-------------------------+

=end text

=for html <p><i>Figure 1</i><img src="data:image/png;base64,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"></img></p>

=head1 SYNTAX

The ditaa syntax L<is documented here|http://ditaa.sourceforge.net/#usage>.

=head1 DEBUGGING

Set the C<DITAA_TRACE> env var and you'll see all of the commands that this
plugin runs printed to C<STDERR>.  If you set the env var to C<keep> the
temporary files referenced in the command will not automatically be deleted, so
you can ensure that the text and image diagrams were created correctly.

=head1 PERL SUPPORT POLICY

Because this module is geared towards helping release code, as opposed to
helping run code, I only aim at supporting the last 3 releases of Perl.  So for
example, at the time of writing that would be 5.22, 5.20, and 5.18.  As an
author who is developing against Perl and using this to release modules, you can
use either L<perlbrew|http://perlbrew.pl/> or
L<plenv|https://github.com/tokuhirom/plenv> to get a more recent perl for
building releases.

Don't bother sending patches to support older versions; I could probably support
5.8 if I wanted, but this is more so that I can continue to use new perl
features.

=head1 AUTHOR

Arthur Axel "fREW" Schmidt <frioux+cpan@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Arthur Axel "fREW" Schmidt.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
